<?php

/**
 * Prescription.class.php
 *
 * @package   OpenEMR
 * @link      https://www.open-emr.org
 * @author    Brady Miller <brady.g.miller@gmail.com>
 * @copyright Copyright (c) 2019 Brady Miller <brady.g.miller@gmail.com>
 * @license   https://github.com/openemr/openemr/blob/master/LICENSE GNU General Public License 3
 */

// Below list of terms are deprecated, but we keep this list
//   to keep track of the official openemr drugs terms and
//   corresponding ID's for reference. Official is referring
//   to the default settings after installing OpenEMR.
//
// define('UNIT_BLANK',0);
// define('UNIT_MG',1);
// define('UNIT_MG_1CC',2);
// define('UNIT_MG_2CC',3);
// define('UNIT_MG_3CC',4);
// define('UNIT_MG_4CC',5);
// define('UNIT_MG_5CC',6);
// define('UNIT_MCG',7);
// define('UNIT_GRAMS',8);
//
// define('INTERVAL_BLANK',0);
// define('INTERVAL_BID',1);
// define('INTERVAL_TID',2);
// define('INTERVAL_QID',3);
// define('INTERVAL_Q_3H',4);
// define('INTERVAL_Q_4H',5);
// define('INTERVAL_Q_5H',6);
// define('INTERVAL_Q_6H',7);
// define('INTERVAL_Q_8H',8);
// define('INTERVAL_QD',9);
// define('INTERVAL_AC',10); // added May 2008
// define('INTERVAL_PC',11); // added May 2008
// define('INTERVAL_AM',12); // added May 2008
// define('INTERVAL_PM',13); // added May 2008
// define('INTERVAL_ANTE',14); // added May 2008
// define('INTERVAL_H',15); // added May 2008
// define('INTERVAL_HS',16); // added May 2008
// define('INTERVAL_PRN',17); // added May 2008
// define('INTERVAL_STAT',18); // added May 2008
//
// define('FORM_BLANK',0);
// define('FORM_SUSPENSION',1);
// define('FORM_TABLET',2);
// define('FORM_CAPSULE',3);
// define('FORM_SOLUTION',4);
// define('FORM_TSP',5);
// define('FORM_ML',6);
// define('FORM_UNITS',7);
// define('FORM_INHILATIONS',8);
// define('FORM_GTTS_DROPS',9);
// define('FORM_CR',10);
// define('FORM_OINT',11);
//
// define('ROUTE_BLANK',0);
// define("ROUTE_PER_ORIS", 1);
// define("ROUTE_PER_RECTUM", 2);
// define("ROUTE_TO_SKIN", 3);
// define("ROUTE_TO_AFFECTED_AREA", 4);
// define("ROUTE_SUBLINGUAL", 5);
// define("ROUTE_OS", 6);
// define("ROUTE_OD", 7);
// define("ROUTE_OU", 8);
// define("ROUTE_SQ", 9);
// define("ROUTE_IM", 10);
// define("ROUTE_IV", 11);
// define("ROUTE_PER_NOSTRIL", 12);
// define("ROUTE_B_EAR", 13);
// define("ROUTE_L_EAR", 14);
// define("ROUTE_R_EAR", 15);
//
// define('SUBSTITUTE_YES',1);
// define('SUBSTITUTE_NO',2);
//


require_once(__DIR__ . "/../lists.inc.php");


/**
 * class Prescription
 *
 */

use OpenEMR\Common\ORDataObject\ORDataObject;
use OpenEMR\Common\Database\QueryUtils;
use OpenEMR\Services\FHIR\Enum\FHIRMedicationIntentEnum;
use OpenEMR\Services\ListService;

class Prescription extends ORDataObject
{
    /**
     *
     * @access public
     */

    /**
     *
     * static
     */
    public $form_array;
    public $unit_array;
    public $route_array;
    public $interval_array;
    public $substitute_array;
    public $medication_array;
    public $refills_array;

    /**
     *
     * @access private
     */

    public $id;
    public $patient = '';
    public $pharmacist = '';
    public $date_added;
    public $txDate;
    public $date_modified;
    public $pharmacy = '';
    public $start_date;
    public $filled_date;
    public $provider = '';
    public $note = '';
    public $drug = '';
    public $rxnorm_drugcode = '';
    public $form = '';
    public $dosage = '';
    public $quantity;
    public $size = '';
    public $unit;
    public $route;
    public $interval = '';
    public $substitute;
    public $refills;
    public $per_refill;
    public $medication;

    public $drug_id;
    public $active;
    public $ntx;

    public $encounter;

    public $created_by;

    public $updated_by;

    private string $erx_source;

    public ?string $diagnosis;

    public ?string $usage_category;

    public ?string $request_intent;

    public ?string $request_intent_title = null;

    public ?string $usage_category_title = null;

    // ===============================================================================================
    // BEGIN AI-GENERATED CODE: Properties to track original values for delayed persistence
    // Generated by Claude.AI to support moving database operations from setters to persist function
    // ===============================================================================================

    /**
     * Track original values before persistence to support delayed database operations
     */
    private $original_drug = null;
    private $original_medication = null;
    private $needs_medication_list_update = false;
    private $needs_drug_update = false;
    private $persisted_values = [];

    // ===============================================================================================
    // END AI-GENERATED CODE: Properties to track original values for delayed persistence
    // ===============================================================================================

    /**
    * Constructor sets all Prescription attributes to their default value
    */

    function __construct($id = "", $_prefix = "")
    {
        $this->route_array = $this->load_drug_attributes('drug_route');
        $this->form_array = $this->load_drug_attributes('drug_form');
        $this->interval_array = $this->load_drug_attributes('drug_interval');
        $this->unit_array = $this->load_drug_attributes('drug_units');

        $this->substitute_array = ["",xl("substitution allowed"),
            xl("do not substitute")];

        $this->medication_array = [0 => xl('No'), 1 => xl('Yes')];

        if (is_numeric($id)) {
            $this->id = $id;
        } else {
            $id = "";
        }

        //$this->unit = UNIT_MG;
        //$this->route = ROUTE_PER_ORIS;
        //$this->quantity = 1;
        //$this->size = 1;
            $this->refills = 0;
        //$this->form = FORM_TABLET;
            $this->substitute = false;
            $this->_prefix = $_prefix;
            $this->_table = "prescriptions";
            $this->pharmacy = new Pharmacy();
            $this->pharmacist = new Person();
            // default provider is the current user
            $this->provider = new Provider($_SESSION['authUserID']);
            $this->patient = new Patient();
            $this->start_date = date("Y-m-d");
            $this->date_added = date("Y-m-d H:i:s");
            $this->date_modified = date("Y-m-d H:i:s");
            $this->created_by = $_SESSION['authUserID'];
            $this->updated_by = $_SESSION['authUserID'];
            $this->per_refill = 0;
            $this->encounter = $_SESSION['encounter'];
            $this->note = "";
            $this->request_intent = FHIRMedicationIntentEnum::ORDER->value;
            $this->usage_category = "outpatient";

            $this->drug_id = 0;
            $this->active = 1;

        $this->ntx = 0;
        $this->diagnosis = "";

        for ($i = 0; $i < 21; $i++) {
            $this->refills_array[$i] = sprintf("%02d", $i);
        }

        if ($id != "") {
            $this->populate();
        }
    }


    // ===============================================================================================
    // BEGIN AI-GENERATED CODE: Refactored persist function with moved database operations
    // Generated by Claude.AI to consolidate all database updates in persist function
    // ===============================================================================================

    function persist()
    {
        $this->date_modified = date("Y-m-d H:i:s");
        if ($this->id == "") {
            $this->date_added = date("Y-m-d H:i:s");
        }


        // these values are not posted so we need to set them here
        $listService = new ListService();
        // need to populate the usage_category_title and request_intent_title before persisting
        $usageCategory = $listService->getListOption('medication-usage-category', $this->usage_category);
        $requestIntent = $listService->getListOption('medication-request-intent', $this->request_intent);
        $this->usage_category_title = $usageCategory['title'] ?? '';
        $this->request_intent_title = $requestIntent['title'] ?? '';
        // Store original prescription ID before potential creation
        $original_prescription_id = $this->id;

        // Perform the main prescription persist operation first
        $result = parent::persist();

        if ($result) {
            // Handle medication list updates that were moved from set_medication()
            if ($this->needs_medication_list_update) {
                $this->handle_medication_list_updates();
            }

            // Handle drug updates that were moved from set_drug()
            if ($this->needs_drug_update) {
                $this->handle_drug_updates();
            }

            // Reset tracking flags after successful persistence
            $this->reset_change_tracking();
        }

        return $result;
    }

    /**
     * Handle medication list updates moved from set_medication function
     */
    private function handle_medication_list_updates()
    {
        global $ISSUE_TYPES;

        // Avoid making a mess if we are not using the "medication" issue type.
        if (isset($ISSUE_TYPES) && !$ISSUE_TYPES['medication']) {
            return;
        }

        //below statements are bypassing the persist() function and being used directly in database statements, hence need to use the functions in library/formdata.inc.php
        // they have already been run through populate() hence stripped of escapes, so now need to be escaped for database (add_escape_custom() function).

        // ===============================================================================================
        // BEGIN AI-GENERATED CODE: Enhanced medication list lookup with prescription_id priority
        // Generated by Claude.AI to first try prescription_id lookup before falling back to title matching
        // ===============================================================================================

        $dataRow = [];

        // First try to find by prescription_id if we have one (more reliable and direct)
        if (!empty($this->id)) {
            $dataRow = QueryUtils::fetchRecords(
                "select l.id from lists l join lists_medication lm on l.id = lm.list_id where "
                . " l.type = 'medication' and l.activity = 1 and (l.enddate is null or cast(now() as date) < l.enddate) and "
                . " lm.prescription_id = ? and l.pid = ? limit 1",
                [$this->id, $this->patient->id]
            );
            $dataRow = $dataRow[0] ?? [];
        }

        // If not found by prescription_id, fall back to title matching (legacy logic)
        if (empty($dataRow['id'])) {
            $dataRow = QueryUtils::fetchRecords("select id from lists where type = 'medication' and "
            . " activity = 1 and (enddate is null or cast(now() as date) < enddate) and "
             . " upper(trim(title)) = upper(trim(?)) and pid = ? limit 1", [$this->drug, $this->patient->id]);
            $dataRow = $dataRow[0] ?? [];
        }

        // ===============================================================================================
        // END AI-GENERATED CODE: Enhanced medication list lookup with prescription_id priority
        // ===============================================================================================

        if ($this->medication && !isset($dataRow['id'])) {
            // ===============================================================================================
            // BEGIN AI-GENERATED CODE: Enhanced inactive medication lookup with prescription_id priority
            // Generated by Claude.AI to first try prescription_id lookup for inactive medications
            // ===============================================================================================

            $inactiveDataRow = [];

            // First try to find inactive medication by prescription_id if we have one
            if (!empty($this->id)) {
                $dataRow = QueryUtils::fetchRecords(
                    "select l.id from lists l join lists_medication lm on "
                    . " l.id = lm.list_id where l.type = 'medication' and l.activity = 0 AND (l.enddate is null or cast(now() as date) < l.enddate) "
                    . " and lm.prescription_id = ? and l.pid = ? limit 1",
                    [$this->id, $this->patient->id]
                );
                $inactiveDataRow = $dataRow[0] ?? [];
            }

            // If not found by prescription_id, fall back to title matching for inactive medications
            if (empty($inactiveDataRow['id'])) {
                $dataRow = QueryUtils::fetchRecords(
                    "select id from lists where type = 'medication' "
                    . "and activity = 0 AND (enddate is null or cast(now() as date) < enddate) and upper(trim(title)) = upper(trim(?)) and pid = ? limit 1",
                    [$this->original_drug, $this->patient->id]
                );
                $inactiveDataRow = $dataRow[0] ?? [];
            }

            // ===============================================================================================
            // END AI-GENERATED CODE: Enhanced inactive medication lookup with prescription_id priority
            // ===============================================================================================

            if (!isset($inactiveDataRow['id'])) {
                //add the record to the medication list
                $medListId = QueryUtils::sqlInsert(
                    "insert into lists(date,begdate,type,activity,pid,user,groupname,title) "
                    . " values (now(),cast(now() as date),'medication',1,?,?,?,?)",
                    [$this->patient->id, $_SESSION['authUser'], $_SESSION['authProvider'], $this->drug]
                );
                $this->gen_lists_medication($medListId);
            } else {
                QueryUtils::sqlStatementThrowException(
                    'update lists set activity = 1,user = ?, groupname = ? where id = ?',
                    [$_SESSION['authUser'], $_SESSION['authProvider'], $inactiveDataRow['id']]
                );
                $this->gen_lists_medication($inactiveDataRow["id"]);
            }
        } elseif (!$this->medication && isset($dataRow['id'])) {
            //remove the drug from the medication list if it exists
            QueryUtils::sqlStatementThrowException(
                'update lists set activity = 0,user = ?, groupname = ? where id = ?',
                [$_SESSION['authUser'], $_SESSION['authProvider'], $dataRow['id']]
            );
        } elseif ($this->medication && isset($dataRow['id'])) {
            $this->gen_lists_medication($dataRow["id"]);
        }
    }

    /**
     * Handle drug updates moved from set_drug function
     */
    private function handle_drug_updates()
    {
        // If the medication already exists in the list and the drug name is being changed, update the title there as well
        if (!empty($this->original_drug) && $this->medication && $this->drug !== $this->original_drug) {
            // ===============================================================================================
            // BEGIN AI-GENERATED CODE: Enhanced drug update lookup with prescription_id priority
            // Generated by Claude.AI to first try prescription_id lookup before falling back to original drug title matching
            // ===============================================================================================

            $dataRow = [];

            // First try to find by prescription_id if we have one (more reliable and direct)
            if (!empty($this->id)) {
                $dataRow = QueryUtils::fetchRecords(
                    "select l.id from lists l join lists_medication lm on "
                    . " l.id = lm.list_id where l.type = 'medication' and (l.enddate is null or cast(now() as date) < l.enddate) "
                    . " and lm.prescription_id = ? and l.pid = ? limit 1",
                    [
                        $this->id, $this->patient->id
                    ]
                );
                $dataRow = $dataRow[0] ?? [];
            }

            // If not found by prescription_id, fall back to original drug title matching (legacy logic)
            if (empty($dataRow['id'])) {
                $dataRow = QueryUtils::fetchRecords(
                    "select id from lists where type = 'medication' and "
                    . "(enddate is null or cast(now() as date) < enddate) and upper(trim(title)) = upper(trim(?)) and pid = ? limit 1",
                    [$this->original_drug, $this->patient->id]
                );
                $dataRow = $dataRow[0] ?? [];
            }

            // ===============================================================================================
            // END AI-GENERATED CODE: Enhanced drug update lookup with prescription_id priority
            // ===============================================================================================

            if (isset($dataRow['id'])) {
                QueryUtils::sqlStatementThrowException(
                    'update lists set activity = 1'
                    . " ,user = ?, groupname = ?, title = ? where id = ?",
                    [$_SESSION['authUser'], $_SESSION['authProvider'], $this->drug, $dataRow['id']]
                );
                $this->gen_lists_medication($dataRow["id"]);
            }
        }
    }

    /**
     * Reset change tracking flags after successful persistence
     */
    private function reset_change_tracking()
    {
        $this->needs_medication_list_update = false;
        $this->needs_drug_update = false;
        $this->original_drug = $this->drug;
        $this->original_medication = $this->medication;
        $this->persisted_values = [
            'drug' => $this->drug,
            'medication' => $this->medication
        ];
    }

    // ===============================================================================================
    // END AI-GENERATED CODE: Refactored persist function with moved database operations
    // ===============================================================================================

    function populate()
    {
        parent::populate();
        // for old historical data we are going to populate our created_by and updated_by
        if (empty($this->created_by)) {
            $this->created_by = $this->get_provider_id();
        }
        if (empty($this->updated_by)) {
            $this->updated_by = $this->get_provider_id();
        }

        // ===============================================================================================
        // BEGIN AI-GENERATED CODE: Initialize tracking values after population
        // Generated by Claude.AI to set baseline values for change tracking
        // ===============================================================================================

        // Initialize original values for change tracking
        $this->original_drug = $this->drug;
        $this->original_medication = $this->medication;
        $this->persisted_values = [
            'drug' => $this->drug,
            'medication' => $this->medication
        ];

        // ===============================================================================================
        // END AI-GENERATED CODE: Initialize tracking values after population
        // ===============================================================================================
    }

    function toString($html = false)
    {
        $string = "\n"
            . "ID: " . $this->id . "\n"
            . "Patient:" . $this->patient . "\n"
            . "Patient ID:" . $this->patient->id . "\n"
            . "Pharmacist: " . $this->pharmacist . "\n"
            . "Pharmacist ID: " . $this->pharmacist->id . "\n"
            . "Date Added: " . $this->date_added . "\n"
            . "Date Modified: " . $this->date_modified . "\n"
            . "Pharmacy: " . $this->pharmacy . "\n"
            . "Pharmacy ID:" . $this->pharmacy->id . "\n"
            . "Start Date: " . $this->start_date . "\n"
            . "Filled Date: " . $this->filled_date . "\n"
            . "Provider: " . $this->provider . "\n"
            . "Provider ID: " . $this->provider->id . "\n"
            . "Note: " . $this->note . "\n"
            . "Drug: " . $this->drug . "\n"
            . "Code: " . $this->rxnorm_drugcode . "\n"
            . "Form: " . $this->form_array[$this->form] . "\n"
            . "Dosage: " . $this->dosage . "\n"
            . "Qty: " . $this->quantity . "\n"
            . "Size: " . $this->size . "\n"
            . "Unit: " . $this->unit_array[$this->unit] . "\n"
            . "Route: " . $this->route_array[$this->route] . "\n"
            . "Interval: " . $this->interval_array[$this->interval] . "\n"
            . "Substitute: " . $this->substitute_array[$this->substitute] . "\n"
            . "Refills: " . $this->refills . "\n"
            . "Per Refill: " . $this->per_refill . "\n"
            . "Drug ID: " . $this->drug_id . "\n"
            . "Active: " . $this->active . "\n"
            . "Transmitted: " . $this->ntx;
        return $html ? nl2br($string) : $string;
    }

    private function load_drug_attributes($id)
    {
        $res = sqlStatement("SELECT * FROM list_options WHERE list_id = ? AND activity = 1 ORDER BY seq", [$id]);
        while ($row = sqlFetchArray($res)) {
            $arr[$row['option_id']] = $row['title'] == '' ? ' ' : xl_list_label($row['title']);
        }

        return $arr;
    }

    function get_encounter()
    {
        // this originally was the $_SESSION['encounter'] which seems really dangerous if a prescription is created when
        // one encounter is open in the session and then the prescription has any updates to the original prescription when another encounter is open in the session
        // so this value is now going to be set when the prescription is created and then remain static for that prescription
        return $this->encounter;
    }

    function get_unit_display($display_form = "")
    {
        return( ($this->unit_array[$this->unit] ?? '') );
    }

    function get_unit()
    {
        return $this->unit;
    }
    function set_unit($unit)
    {
        if (is_numeric($unit)) {
            $this->unit = $unit;
        }
    }

    function set_id($id)
    {
        if (!empty($id) && is_numeric($id)) {
            $this->id = $id;
        }
    }
    function get_id()
    {
        return $this->id;
    }

    function get_dosage_display($display_form = "")
    {
        if (empty($this->form) && empty($this->interval)) {
            return( $this->dosage );
        } else {
            return ($this->dosage . " " . xl('in') . " " . ($this->form_array[$this->form] ?? '') . " " . ($this->interval_array[$this->interval] ?? ''));
        }
    }

    function set_dosage($dosage)
    {
        $this->dosage = $dosage;
    }
    function get_dosage()
    {
        return $this->dosage;
    }

    function set_form($form)
    {
        if (is_numeric($form)) {
            $this->form = $form;
        }
    }
    function get_form()
    {
        return $this->form;
    }

    function set_refills($refills)
    {
        if (is_numeric($refills)) {
            $this->refills = $refills;
        }
    }
    function get_refills()
    {
        return $this->refills;
    }

    function set_size($size)
    {
        $this->size = preg_replace("/[^0-9\/\.\-]/", "", (string) $size);
    }
    function get_size()
    {
        return $this->size;
    }

    function set_quantity($qty)
    {
        $this->quantity = $qty;
    }
    function get_quantity()
    {
        return $this->quantity;
    }

    function set_route($route)
    {
        $this->route = $route;
    }
    function get_route()
    {
        return $this->route;
    }

    function set_interval($interval)
    {
        if (is_numeric($interval)) {
            $this->interval = $interval;
        }
    }
    function get_interval()
    {
        return $this->interval;
    }

    function set_substitute($sub)
    {
        if (is_numeric($sub)) {
            $this->substitute = $sub;
        }
    }
    function get_substitute()
    {
        return $this->substitute;
    }
    function set_erx_source($erx_source)
    {
        $this->erx_source = $erx_source;
    }
    public function gen_lists_medication($list_id): void
    {
        $instructions = $this->size . ($this->unit_array[$this->unit] ?? '') . "\t\t" . $this->get_dosage_display();
        if (!empty($list_id)) {
            $listsMedication = QueryUtils::fetchRecords("select prescription_id,list_id from lists_medication where list_id = ? limit 1", [$list_id]);
            if (!empty($listsMedication)) {
                $bind = [$instructions, $this->usage_category ?? '', $this->usage_category_title ?? ''
                    , $this->request_intent ?? '', $this->request_intent_title ?? ''];
                $sql = "update lists_medication set drug_dosage_instructions = ? "
                    . ", usage_category=?, usage_category_title = ?, request_intent=?, request_intent_title =? ";
                // old data may not have prescription_id set so we need to link it here if it has been matched previously
                if (empty($listsMedication[0]['prescription_id'])) {
                    $sql .= ", prescription_id = ? ";
                    $bind[] = $this->id;
                }
                $sql .= " where list_id = ? ";
                $bind[] = $list_id;
                QueryUtils::sqlStatementThrowException($sql, $bind);
            } else {
                QueryUtils::sqlInsert("insert into lists_medication(list_id, drug_dosage_instructions, prescription_id"
                . ",usage_category, usage_category_title, request_intent,  request_intent_title ) values (?, ?, ?, ?, ?, ?, ?)", [$list_id, $instructions, $this->id
                    , $this->usage_category ?? '', $this->usage_category_title ?? ''
                    , $this->request_intent ?? '', $this->request_intent_title ?? '']);
            }
        }
    }
    // ===============================================================================================
    // BEGIN AI-GENERATED CODE: Refactored set_medication to remove database operations
    // Generated by Claude.AI to move database logic to persist function while tracking changes
    // ===============================================================================================

    public function set_medication($med): void
    {
        // Store original value if not already tracked
        if ($this->original_medication === null) {
            $this->original_medication = $this->medication;
        }

        // Update the property value
        $this->medication = $med;

        // Mark that medication list needs updating during persist
        $this->needs_medication_list_update = true;
    }

    // ===============================================================================================
    // END AI-GENERATED CODE: Refactored set_medication to remove database operations
    // ===============================================================================================

    function get_medication()
    {
        return $this->medication;
    }

    function set_per_refill($pr)
    {
        if (is_numeric($pr)) {
            $this->per_refill = $pr;
        }
    }
    function get_per_refill()
    {
        return $this->per_refill;
    }

    function set_patient_id($id)
    {
        if (is_numeric($id)) {
            $this->patient = new Patient($id);
        }
    }
    function get_patient_id()
    {
        return $this->patient->id;
    }

    function set_provider_id($id)
    {
        if (is_numeric($id)) {
            $this->provider = new Provider($id);
        }
    }
    function get_provider_id()
    {
        return $this->provider->id;
    }

    function set_created_by($id)
    {
        if (is_numeric($id)) {
            $this->created_by = $id;
        }
    }
    function get_created_by()
    {
        return $this->created_by;
    }

    function set_updated_by($id)
    {
        if (is_numeric($id)) {
            $this->updated_by = $id;
        }
    }
    function get_updated_by()
    {
        return $this->updated_by;
    }

    public function get_diagnosis(): ?string
    {
        return $this->diagnosis;
    }

    public function set_diagnosis(array|string $diagnosis): void
    {
        // codes are concatenated with a ';' if multiple
        if (is_array($diagnosis)) {
            $diagnosis = implode(';', $diagnosis);
        }
        $this->diagnosis = $diagnosis;
    }

    public function set_request_intent(?string $intent): void
    {
        $this->request_intent = $intent;
    }

    public function get_request_intent(): ?string
    {
        return $this->request_intent;
    }

    public function set_usage_category_title(?string $title): void
    {
        $this->usage_category_title = $title;
    }

    public function get_usage_category_title(): ?string
    {
        return $this->usage_category_title;
    }

    public function set_request_intent_title(?string $title): void
    {
        $this->request_intent_title = $title;
    }

    public function get_request_intent_title(): ?string
    {
        return $this->request_intent_title;
    }

    public function get_usage_category(): ?string
    {
        return $this->usage_category;
    }

    public function set_usage_category(?string $category): void
    {
        $this->usage_category = $category;
    }

    function set_provider($pobj)
    {
        if ($pobj::class == "provider") {
            $this->provider = $pobj;
        }
    }

    function set_pharmacy_id($id)
    {
        if (is_numeric($id)) {
            $this->pharmacy = new Pharmacy($id);
        }
    }
    function get_pharmacy_id()
    {
        return $this->pharmacy->id;
    }

    function set_pharmacist_id($id)
    {
        if (is_numeric($id)) {
            $this->pharmacist = new Person($id);
        }
    }
    function get_pharmacist()
    {
        return $this->pharmacist->id;
    }

    function get_start_date_y()
    {
        $ymd = explode("-", (string) $this->start_date);
        return $ymd[0];
    }
    function set_start_date_y($year)
    {
        if (is_numeric($year)) {
            $ymd = explode("-", (string) $this->start_date);
            $ymd[0] = $year;
            $this->start_date = $ymd[0] . "-" . $ymd[1] . "-" . $ymd[2];
        }
    }
    function get_start_date_m()
    {
        $ymd = explode("-", (string) $this->start_date);
        return $ymd[1];
    }
    function set_start_date_m($month)
    {
        if (is_numeric($month)) {
            $ymd = explode("-", (string) $this->start_date);
            $ymd[1] = $month;
            $this->start_date = $ymd[0] . "-" . $ymd[1] . "-" . $ymd[2];
        }
    }
    function get_start_date_d()
    {
        $ymd = explode("-", (string) $this->start_date);
        return $ymd[2];
    }
    function set_start_date_d($day)
    {
        if (is_numeric($day)) {
            $ymd = explode("-", (string) $this->start_date);
            $ymd[2] = $day;
            $this->start_date = $ymd[0] . "-" . $ymd[1] . "-" . $ymd[2];
        }
    }
    function get_start_date()
    {
        return $this->start_date;
    }
    function set_start_date($date)
    {
        return $this->start_date = $date;
    }

    // TajEmo work by CB 2012/05/30 01:56:32 PM added encounter for auto ticking of checkboxes
    function set_encounter($enc)
    {
        return $this->encounter = $enc;
    }

    function get_date_added()
    {
        return $this->date_added;
    }
    function set_date_added($date)
    {
        return $this->date_added = $date;
    }
    function set_txDate($txdate)
    {
        return $this->txDate = $txdate;
    }
    function get_txDate()
    {
        return $this->txDate;
    }

    function get_date_modified()
    {
        return $this->date_modified;
    }
    function set_date_modified($date)
    {
        return $this->date_modified = $date;
    }

    function get_filled_date()
    {
        return $this->filled_date;
    }
    function set_filled_date($date)
    {
        return $this->filled_date = $date;
    }

    function set_note($note)
    {
        $this->note = $note;
    }

    public function set_drug_dosage_instructions(?string $instructions): void
    {
        $this->note = $instructions;
    }

    function get_note()
    {
        return $this->note;
    }

    public function set_drug($drug): void
    {
        // Store original value if not already tracked
        if ($this->original_drug === null) {
            $this->original_drug = $this->drug;
        }

        // Update the property value
        $this->drug = $drug;

        // Mark that drug updates are needed during persist
        $this->needs_drug_update = true;
    }
    function get_drug()
    {
        return $this->drug;
    }
    function set_ntx($ntx)
    {
        $this->ntx = $ntx;
    }
    function get_ntx()
    {
        return $this->ntx;
    }

    function set_rxnorm_drugcode($rxnorm_drugcode)
    {
        $this->rxnorm_drugcode = $rxnorm_drugcode;
    }
    function get_rxnorm_drugcode()
    {
        return $this->rxnorm_drugcode;
    }

    function get_filled_by_id()
    {
        return $this->pharmacist->id;
    }
    function set_filled_by_id($id)
    {
        if (is_numeric($id)) {
            return $this->pharmacist->id = $id;
        }
    }

    function set_drug_id($drug_id)
    {
        $this->drug_id = $drug_id;
    }
    function get_drug_id()
    {
        return $this->drug_id;
    }

    function set_active($active)
    {
        $this->active = $active;
    }
    function get_active()
    {
        return $this->active;
    }
    function get_prescription_display()
    {
        $pconfig = $GLOBALS['oer_config']['prescriptions'];

        switch ($pconfig['format']) {
            case "FL":
                return $this->get_prescription_florida_display();
                break;
            default:
                break;
        }

        $sql = "SELECT * FROM users JOIN facility AS f ON f.name = users.facility where users.id ='" . add_escape_custom($this->provider->id) . "'";
        $db = get_db();
        $results = $db->Execute($sql);
        if (!$results->EOF) {
            $string = $results->fields['name'] . "\n"
                    . $results->fields['street'] . "\n"
                    . $results->fields['city'] . ", " . $results->fields['state'] . " " . $results->fields['postal_code'] . "\n"
                    . $results->fields['phone'] . "\n\n";
        }

        $string .= ""
                . "Prescription For:" . "\t" . $this->patient->get_name_display() . "\n"
                . "DOB:" . "\t" . $this->patient->get_dob() . "\n"
                . "Start Date: " . "\t\t" . $this->start_date . "\n"
                . "Provider: " . "\t\t" . $this->provider->get_name_display() . "\n"
                . "Provider DEA No.: " . "\t\t" . $this->provider->federal_drug_id . "\n"
                . "Drug: " . "\t\t\t" . $this->drug . "\n"
                . "Dosage: " . "\t\t" . $this->dosage . " in " . ($this->form_array[$this->form] ?? '') . " form " . ($this->interval_array[$this->interval] ?? '') . "\n"
                . "Qty: " . "\t\t\t" . $this->quantity . "\n"
                . "Medication Unit: " . "\t" . $this->size  . " " . ($this->unit_array[$this->unit] ?? '') . "\n"
                . "Substitute: " . "\t\t" . $this->substitute_array[$this->substitute] . "\n";
        if ($this->refills > 0) {
            $string .= "Refills: " . "\t\t" . $this->refills . ", of quantity: " . $this->per_refill . "\n";
        }

        $string .= "\n" . "Notes: \n" . $this->note . "\n";
        return $string;
    }

    function get_prescription_florida_display()
    {

        $db = get_db();
        $ntt = new NumberToText($this->quantity);
        $ntt2 = new NumberToText($this->per_refill);
        $ntt3 = new NumberToText($this->refills);

        $string = "";

        $gnd = $this->provider->get_name_display();

        while (strlen((string) $gnd) < 31) {
            $gnd .= " ";
        }

        $string .= $gnd . $this->provider->federal_drug_id . "\n";

        $sql = "SELECT * FROM users JOIN facility AS f ON f.name = users.facility where users.id ='" . add_escape_custom($this->provider->id) . "'";
        $results = $db->Execute($sql);

        if (!$results->EOF) {
            $rfn = $results->fields['name'];

            while (strlen((string) $rfn) < 31) {
                $rfn .= " ";
            }

            $string .= $rfn . $this->provider->get_provider_number_default() . "\n"
                    . $results->fields['street'] . "\n"
                    . $results->fields['city'] . ", " . $results->fields['state'] . " " . $results->fields['postal_code'] . "\n"
                    . $results->fields['phone'] . "\n";
        }

        $string .= "\n";
        $string .= strtoupper((string) $this->patient->lname) . ", " . ucfirst((string) $this->patient->fname) . " " . $this->patient->mname . "\n";
        $string .= "DOB " .  $this->patient->date_of_birth . "\n";
        $string .= "\n";
        $string .= date("F j, Y", strtotime((string) $this->start_date)) . "\n";
        $string .= "\n";
        $string .= strtoupper((string) $this->drug) . " " . $this->size  . " " . ($this->unit_array[$this->unit] ?? '') . "\n";
        if (strlen((string) $this->note) > 0) {
            $string .= "Notes: \n" . $this->note . "\n";
        }

        if (!empty($this->dosage)) {
            $string .= $this->dosage;
            if (!empty($this->form)) {
                $string .= " " . $this->form_array[$this->form];
            }

            if (!empty($this->interval)) {
                $string .= " " . $this->interval_array[$this->interval];
            }

            if (!empty($this->route)) {
                $string .= " " . $this->route_array[$this->route] . "\n";
            }
        }

        if (!empty($this->quantity)) {
            $string .= "Disp: " . $this->quantity . " (" . trim(strtoupper($ntt->convert())) . ")" . "\n";
        }

        $string .= "\n";
        $string .= "Refills: " . $this->refills . " (" . trim(strtoupper($ntt3->convert())) . "), Per Refill Disp: " . $this->per_refill . " (" . trim(strtoupper($ntt2->convert())) . ")" . "\n";
        $string .= $this->substitute_array[$this->substitute] . "\n";
        $string .= "\n";

        return $string;
    }

    static function prescriptions_factory(
        $patient_id,
        $order_by = "active DESC, date_modified DESC, date_added DESC"
    ) {

        $prescriptions = [];
        $p = new Prescription();
        $sql = "SELECT id FROM " . escape_table_name($p->_table) . " WHERE patient_id = ? " .
                "ORDER BY " . add_escape_custom($order_by);
        $results = sqlQ($sql, [$patient_id]);
        while ($row = sqlFetchArray($results)) {
            $prescriptions[] = new Prescription($row['id']);
        }

        return $prescriptions;
    }

    function get_dispensation_count()
    {
        if (empty($this->id)) {
            return 0;
        }

        $refills_row = sqlQuery("SELECT count(*) AS count FROM drug_sales " .
                    "WHERE prescription_id = ? AND quantity > 0", [$this->id]);
        return $refills_row['count'];
    }
}// end of Prescription
