<?php

/**
 * CustomClientCredentialsGrant implements the requirements for ONC SMART Bulk FHIR Client Credentials grant.
 * @package openemr
 * @link      http://www.open-emr.org
 * @author    Stephen Nielson <stephen@nielson.org>
 * @copyright Copyright (c) 2021 Stephen Nielson <stephen@nielson.org>
 * @license   https://github.com/openemr/openemr/blob/master/LICENSE GNU General Public License 3
 */

namespace OpenEMR\Common\Auth\OpenIDConnect\Grant;

use DateInterval;
use League\OAuth2\Server\Entities\ClientEntityInterface;
use League\OAuth2\Server\Entities\ScopeEntityInterface;
use League\OAuth2\Server\Exception\OAuthServerException;
use League\OAuth2\Server\Grant\ClientCredentialsGrant;
use League\OAuth2\Server\RequestEvent;
use OpenEMR\Common\Auth\OpenIDConnect\Entities\ClientEntity;
use OpenEMR\Common\Logging\SystemLoggerAwareTrait;
use OpenEMR\Services\JWTClientAuthenticationService;
use OpenEMR\Services\TrustedUserService;
use OpenEMR\Services\UserService;
use Psr\Http\Message\ServerRequestInterface;
use Symfony\Component\HttpFoundation\Session\SessionInterface;

class CustomClientCredentialsGrant extends ClientCredentialsGrant
{
    use SystemLoggerAwareTrait;

    /**
     * @var TrustedUserService
     */
    private readonly TrustedUserService $trustedUserService;

    /**
     * @var UserService
     */
    private UserService $userService;

    /**
     * @var JWTClientAuthenticationService
     */
    private JWTClientAuthenticationService $jwtAuthService;

    /**
     * The required value for the jwt assertion type
     */
    const OAUTH_JWT_CLIENT_ASSERTION_TYPE = 'urn:ietf:params:oauth:client-assertion-type:jwt-bearer';
    /**
     * CustomClientCredentialsGrant constructor.
     * @param SessionInterface $session
     * @param string $authTokenUrl The OAUTH2 token issuing url to be used as the audience parameter for JWT validation
     */
    public function __construct(
        private readonly SessionInterface $session,
        private readonly string $authTokenUrl
    ) {
        $this->trustedUserService = new TrustedUserService();
        $this->userService = new UserService();
    }

    /**
     * Set the JWT authentication service
     *
     * @param JWTClientAuthenticationService $jwtAuthService
     */
    public function setJWTAuthenticationService(JWTClientAuthenticationService $jwtAuthService): void
    {
        $this->jwtAuthService = $jwtAuthService;
    }

    /**
     * @return UserService
     */
    public function getUserService(): UserService
    {
        return $this->userService;
    }

    /**
     * @param UserService $userService
     */
    public function setUserService(UserService $userService): void
    {
        $this->userService = $userService;
    }

    /**
     * We issue an access token, but we force the user account to be our OpenEMR API system user.  We also save off the
     * grant as a TrustedUser which we can use later for revocation if necessary.
     * @param DateInterval $accessTokenTTL
     * @param ClientEntityInterface $client
     * @param string|null $userIdentifier
     * @param array $scopes
     * @return \League\OAuth2\Server\Entities\AccessTokenEntityInterface
     * @throws OAuthServerException If there is a server error, or some other oauth2 violation
     * @throws \League\OAuth2\Server\Exception\UniqueTokenIdentifierConstraintViolationException
     */
    protected function issueAccessToken(DateInterval $accessTokenTTL, ClientEntityInterface $client, $userIdentifier, array $scopes = [])
    {
        // let's grab our user id here.
        if ($userIdentifier === null) {
            // we want to grab our system user
            $systemUser = $this->userService->getSystemUser();
            if (empty($systemUser['uuid'])) {
                $this->getSystemLogger()->error("SystemUser was missing.  System is not setup properly");
                throw OAuthServerException::serverError("Server was not properly setup");
            }
            $userIdentifier = $systemUser['uuid'] ?? null;
        }
        $accessToken = parent::issueAccessToken($accessTokenTTL, $client, $userIdentifier, $scopes); // TODO: Change the autogenerated stub

        // gotta remove since binary and will break json_encode (not used for password granttype, so ok to remove)
        $this->session->remove('csrf_private_key');

        $session_cache = json_encode($this->session->all(), JSON_THROW_ON_ERROR);
        $code = null; // code is used only in authorization_code grant types

        $scopeList = [];

        foreach ($scopes as $scope) {
            if ($scope instanceof ScopeEntityInterface) {
                $scopeList[] = $scope->getIdentifier();
            }
        }

        // we can't get past the api dispatcher without having a trusted user.
        $this->trustedUserService->saveTrustedUser(
            $client->getIdentifier(),
            $userIdentifier,
            implode(" ", $scopeList),
            0,
            $code,
            $session_cache,
            $this->getIdentifier()
        );

        return $accessToken;
    }

    /**
     * Gets the client credentials from the request from the request body or
     * the Http Basic Authorization header
     *
     * @param ServerRequestInterface $request
     *
     * @return array
     */
    protected function getClientCredentials(ServerRequestInterface $request): array
    {
        $logger = $this->getSystemLogger();
        // Check if JWT authentication service is available and request has JWT assertion
        if (!isset($this->jwtAuthService)) {
            $logger->error('CustomClientCredentialsGrant::getClientCredentials: JWT authentication service is not set up');
            throw OAuthServerException::invalidRequest("client_assertion_type", "JWT authentication service is not configured");
        }
        if ($this->jwtAuthService->hasJWTClientAssertion($request)) {
            $logger->debug('CustomClientCredentialsGrant::getClientCredentials: Detected JWT client assertion, using asymmetric authentication');

            try {
                // Extract client ID from JWT
                $clientId = $this->jwtAuthService->extractClientIdFromJWT($request);
                $logger->debug("CustomClientCredentialsGrant::getClientCredentials: Extracted client ID from JWT", ['client_id' => $clientId]);
            } catch (OAuthServerException $e) {
                $logger->error(
                    'CustomClientCredentialsGrant::getClientCredentials: Failed to extract client ID from JWT',
                    ['error' => $e->getMessage(), 'hint' => $e->getHint()]
                );
                throw $e;
            }
            return [$clientId, null]; // No client secret for JWT authentication
        } else {
            throw OAuthServerException::invalidRequest("client_assertion_type", "assertion type is not supported");
        }
    }

    /**
     * Validate the client against the client's jwks
     * @see https://tools.ietf.org/html/rfc7523#section-3
     *
     * @param ServerRequestInterface $request
     *
     * @throws OAuthServerException
     *
     * @return ClientEntityInterface
     */
    protected function validateClient(ServerRequestInterface $request)
    {
        // skip everything else for now.
        [$clientId] = $this->getClientCredentials($request);


        // grab the client
        $client = $this->getClientEntityOrFail($clientId, $request);

        // Currently all the JWK validation stuff is centralized in this
        // grant... but knowledge of the client entity is inside the ClientRepository, either way I don't like the
        // class cohesion problems this creates.
        if (!($client instanceof ClientEntity)) {
            $this->getSystemLogger()->error("CustomClientCredentialsGrant->validateClient() client returned was not a valid ClientEntity ", ['client' => $clientId]);
            throw OAuthServerException::invalidClient($request);
        }
        // validate everything to do with the JWT...
        // Check if JWT authentication service is available and request has JWT assertion
        if (!isset($this->jwtAuthService)) {
            $this->getSystemLogger()->error('CustomClientCredentialsGrant::getClientCredentials: JWT authentication service is not set up');
            throw OAuthServerException::invalidRequest("client_assertion_type", "JWT authentication service is not configured");
        }
        if ($this->jwtAuthService->hasJWTClientAssertion($request)) {
            // Validate JWT assertion
            try {
                $this->jwtAuthService->validateJWTClientAssertion($request, $client);
                $this->getSystemLogger()->debug("CustomClientCredentialsGrant->validateClient() JWT assertion validated successfully");
            } catch (OAuthServerException $e) {
                $this->getSystemLogger()->error(
                    "CustomClientCredentialsGrant->validateClient() JWT validation failed",
                    ['error' => $e->getMessage(), 'hint' => $e->getHint()]
                );
                throw $e;
            }
        } else {
            // we only support JWT client assertions for this grant type
            $this->getSystemLogger()->error("CustomClientCredentialsGrant->validateClient() missing or invalid JWT client assertion", ['client' => $clientId]);
            throw OAuthServerException::invalidClient($request);
        }

        if ($this->clientRepository->validateClient($clientId, null, $this->getIdentifier()) === false) {
            $this->getEmitter()->emit(new RequestEvent(RequestEvent::CLIENT_AUTHENTICATION_FAILED, $request));

            throw OAuthServerException::invalidClient($request);
        }

        // If a redirect URI is provided ensure it matches what is pre-registered
        $redirectUri = $this->getRequestParameter('redirect_uri', $request, null);

        if ($redirectUri !== null) {
            $this->validateRedirectUri($redirectUri, $client, $request);
        }

        return $client;
    }
}
