<?php

/**
 * @package   OpenEMR
 * @link      http://www.open-emr.org
 * @author    Ken Chapple <ken@mi-squared.com>
 * @copyright Copyright (c) 2021 Ken Chapple <ken@mi-squared.com>
 * @license   https://github.com/openemr/openemr/blob/master/LICENSE GNU GeneralPublic License 3
 */

namespace OpenEMR\Services\Qdm;

use OpenEMR\Cqm\Generator;
use OpenEMR\Services\CodeTypesService;
use OpenEMR\Services\Qdm\Interfaces\QdmRequestInterface;
use OpenEMR\Services\Qdm\Interfaces\QdmServiceInterface;
use OpenEMR\Services\Qdm\Services\AllergyIntoleranceService;
use OpenEMR\Services\Qdm\Services\AssessmentService;
use OpenEMR\Services\Qdm\Services\DeviceAppliedService;
use OpenEMR\Services\Qdm\Services\DeviceOrderService;
use OpenEMR\Services\Qdm\Services\DeviceRecommendedService;
use OpenEMR\Services\Qdm\Services\DiagnosisService;
use OpenEMR\Services\Qdm\Services\DiagnosticStudyOrderedService;
use OpenEMR\Services\Qdm\Services\DiagnosticStudyService;
use OpenEMR\Services\Qdm\Services\EncounterService;
use OpenEMR\Services\Qdm\Services\ImmunizationAdministeredService;
use OpenEMR\Services\Qdm\Services\InterventionOrderedService;
use OpenEMR\Services\Qdm\Services\InterventionService;
use OpenEMR\Services\Qdm\Services\LaboratoryTestOrderedService;
use OpenEMR\Services\Qdm\Services\LaboratoryTestService;
use OpenEMR\Services\Qdm\Services\MedicationActiveService;
use OpenEMR\Services\Qdm\Services\SubstanceOrderService;
use OpenEMR\Services\Qdm\Services\PatientService;
use OpenEMR\Services\Qdm\Services\PhysicalExamService;
use OpenEMR\Services\Qdm\Services\ProcedureRecommendedService;
use OpenEMR\Services\Qdm\Services\ProcedureService;
use OpenEMR\Services\Qdm\Services\SubstanceRecommendedService;
use OpenEMR\Services\Qdm\Services\SymptomService;

class QdmBuilder
{
    protected $services = [
        AllergyIntoleranceService::class,
        AssessmentService::class,
        DeviceAppliedService::class,
        DeviceOrderService::class,
        DeviceRecommendedService::class,
        DiagnosisService::class,
        DiagnosticStudyService::class,
        DiagnosticStudyOrderedService::class,
        EncounterService::class,
        ImmunizationAdministeredService::class,
        InterventionService::class,
        InterventionOrderedService::class,
        LaboratoryTestService::class,
        LaboratoryTestOrderedService::class,
        MedicationActiveService::class,
        PhysicalExamService::class,
        ProcedureService::class,
        ProcedureRecommendedService::class,
        SubstanceOrderService::class,
        SubstanceRecommendedService::class,
        SymptomService::class,
    ];

    public function build(QdmRequestInterface $request): array
    {
        // Create the patient service
        $patientService = new PatientService($request, new CodeTypesService());

        // Query all patients and build QDM models in a loop, storing the QDM patient model in an associative array,
        // so we can later look up by PID
        $qdm_patients_map = [];
        $patientResult = $patientService->executeQuery();
        while ($patient = sqlFetchArray($patientResult)) {
            $qdmRecord = new QdmRecord($patient, $patient['pid'], $patient['pid']);
            $qdmPatient = $patientService->makeQdmModel($qdmRecord);
            $qdm_patients_map[$patient['pid']] = $qdmPatient;
        }

        $entityCount = 10000;
        foreach ($this->services as $serviceClass) {
            // Create our service and make sure it implements required methods (inherits from AbstractQdmService).
            $service = new $serviceClass($request, new CodeTypesService());
            if ($service instanceof QdmServiceInterface) {
                // Using the services, query all records for ALL patients in our QdmRequest object, which means we
                // get all relevant models for all patients in one query for this particular service category.
                $result = $service->executeQuery();
                while ($record = sqlFetchArray($result)) {
                    // Create a QDM record with the result. This makes sure we have required PID.
                    if (
                        !isset($record['pid'])
                    ) {
                        throw new \Exception("The query result generated by QdmService::getSqlStatement() must contain a pid");
                    }
                    $qdmRecord = new QdmRecord($record, $record['pid'], $entityCount);
                    // Use the service to make a QDM model with the data from the query result
                    try {
                        $qdmModel = $service->makeQdmModel($qdmRecord);
                        // If for some reason the model doesn't need to return a value, or the date is invalid, makeQdmModel can return null
                        if (!empty($qdmModel)) {
                            // Using the PID map, find the patient this model belongs to and add this data element
                            // to the correct patient's QDM model
                            $qdmPatient = $qdm_patients_map[$qdmRecord->getPid()];
                            if (($qdmPatient ?? null) !== null) {
                                $qdmPatient->add_data_element($qdmModel);
                            } else {
                                // If we don't have a QDM Patient model for this PID it's usually from a patient delete data model leftovers
                                // care plans, medications etc. that were not deleted.
                                error_log("QDM Builder Warning: No QDM Patient model found  on `$serviceClass` for PID = `{$qdmRecord->getPid()}`... Continuing execution.");
                            }
                        } else {
                            error_log("QDM Builder Warning: NULL returned by makeQdmModel() on `$serviceClass` for PID = `{$qdmRecord->getPid()}`... Continuing execution.");
                        }
                    } catch (\Exception $e) {
                        // There was an error creating the model, such as passing a parameter that is not a member of a QDM Object
                        // TODO improve error handling
                        error_log($e->getMessage());
                    }

                    $entityCount++;
                }
            } else {
                throw new \Exception("Service does not implement required contract for making QDM models");
            }
        }

        // Take just the map of models, re-index into simple array without PID as index
        $models = array_values($qdm_patients_map);
        return $models;
    }

    public function _action_generate_models(): never
    {
        $generator = new Generator();
        $generator->execute();
        echo '';
        exit;
    }
}
