<?php

/**
 * TokenSearchField represents a token data type that contains both a system and a code.
 * @package openemr
 * @link      http://www.open-emr.org
 * @author    Stephen Nielson <stephen@nielson.org>
 * @copyright Copyright (c) 2021 Stephen Nielson <stephen@nielson.org>
 * @license   https://github.com/openemr/openemr/blob/master/LICENSE GNU General Public License 3
 */

namespace OpenEMR\Services\Search;

use OpenEMR\Services\Search\SearchFieldType;

class TokenSearchField extends BasicSearchField
{
    /**
     * @param mixed $field
     * @param mixed $values
     * @param bool $isUUID True if the token represents a UUID that is a binary field in the database
     */
    public function __construct(
        $field,
        $values,
        private $isUUID = false
    ) {
        parent::__construct($field, SearchFieldType::TOKEN, $field, $values);
    }

    public function isUuid()
    {
        return $this->isUUID;
    }

    public function addValue(TokenSearchValue $value)
    {
        $values = $this->getValues();
        $values[] = $value;
        $this->setValues($values);
    }

    public function hasCodeValue($code, $system = null)
    {
        $checkSystem = $system !== null;
        foreach ($this->getValues() as $tokenValue) {
            if ($tokenValue instanceof TokenSearchValue && $code == $tokenValue->getCode()) {
                if ($checkSystem) {
                    return $system == $tokenValue->getSystem();
                }
                return true;
            }
        }
        return false;
    }

    public function transformValues(callable $transformer)
    {
        if (!is_callable($transformer)) {
            throw new \BadMethodCallException("transformer function must be callable");
        }
        $values = $this->getValues() ?? [];
        $values = array_map($transformer, $values);
        $this->setValues($values);
        return $values;
    }

    public function setValues(array $values)
    {
        $convertedFields = [];

        foreach ($values as $value) {
            if ($value instanceof TokenSearchValue) {
                $convertedFields[] = $value;
                continue;
            }

            $convertedFields[] = $this->createTokenSearchValue($value);
        }
        return parent::setValues($convertedFields);
    }

    private function createTokenSearchValue($value)
    {
        if (!is_string($value)) {
            throw new \InvalidArgumentException("Token value must be a valid string");
        }

        return TokenSearchValue::buildFromFHIRString($value, $this->isUUID);
    }

    /**
     * Checks if this TokenSearchField contains all of the values of the given TokenSearchField
     * a token value is considered contained if both system and code match OR if only code matches and the system
     * of the containedSearchToken is null
     * @param TokenSearchField $containedSearchToken
     * @return bool
     */
    public function containsSearchToken(TokenSearchField $containedSearchToken): bool
    {
        foreach ($containedSearchToken->getValues() as $containedValue) {
            $found = false;
            foreach ($this->getValues() as $ownValue) {
                if ($containedValue instanceof TokenSearchValue && $ownValue instanceof TokenSearchValue) {
                    if ($containedValue->getCode() === $ownValue->getCode()) {
                        // codes match, now check system
                        if ($containedValue->getSystem() === null || $containedValue->getSystem() === $ownValue->getSystem()) {
                            $found = true;
                            break;
                        }
                    }
                }
            }
            if (!$found) {
                return false;
            }
        }
        return true;
    }
}
