<?php

/**
 * GeoTelemetry class
 *
 * Over 80% of GeoTelemetry class code was generated by ChatGPT code copilot.
 * The rest was modified to fit the OpenEMR coding style.
 *
 * Lightweight geolocation and anonymization utility
 * - Uses public IP lookup APIs
 * - Composer-free
 * - Fallback support and IP anonymization
 *
 * @package        OpenEMR
 * @link           https://www.open-emr.org
 * @link           https://opencoreemr.com
 * @author         Jerry Padgett <sjpadgett@gmail.com>
 * @author         Michael A. Smith <michael@opencoreemr.com>
 * @copyright      Copyright (c) 2025 <sjpadgett@gmail.com>
 * @copyright      Copyright (c) 2026 OpenCoreEMR Inc.
 * @license        https://github.com/openemr/openemr/blob/master/LICENSE GNU General Public License 3
 */

namespace OpenEMR\Telemetry;

use OpenEMR\Common\Utils\ValidationUtils;

class GeoTelemetry implements GeoTelemetryInterface
{
    private const GET_IP_URL = 'https://api.ipify.org';

    /**
     * Anonymize IP using SHA-256 hashing
     */
    public function anonymizeIp(string $ip): string
    {
        return hash('sha256', $ip);
    }

    /**
     * Get geolocation data from IP using a lightweight external API
     */
    public function getGeoData(string $ip): array
    {
        $data = $this->fetchJson("https://ipapi.co/{$ip}/json/");
        if (isset($data['country_name'])) {
            return [
                'country' => $data['country_name'] ?? null,
                'region' => $data['region'] ?? null,
                'city' => $data['city'] ?? null,
                'latitude' => $data['latitude'] ?? null,
                'longitude' => $data['longitude'] ?? null,
            ];
        }
        // fallback to geoplugin.net
        $data = $this->fetchJson("http://www.geoplugin.net/json.gp?ip={$ip}");
        if (isset($data['geoplugin_countryName'])) {
            return [
                'country' => $data['geoplugin_countryName'] ?? null,
                'region' => $data['geoplugin_region'] ?? null,
                'city' => $data['geoplugin_city'] ?? null,
                'latitude' => $data['geoplugin_latitude'] ?? null,
                'longitude' => $data['geoplugin_longitude'] ?? null,
            ];
        }

        return ['error' => 'IP lookup failed'];
    }

    /**
     * Get geolocation of the current server (public-facing IP)
     */
    public function getServerGeoData(): array
    {
        $ip = trim($this->fetchText(self::GET_IP_URL));
        if (ValidationUtils::isValidIpAddress($ip)) {
            return $this->getGeoData($ip);
        }
        return ['error' => 'Unable to determine server IP'];
    }

    /**
     * Get raw text data from a given URL
     *
     * @param string $url
     * @return string
     */
    protected function fetchText(string $url): string
    {
        $ctx = stream_context_create(['http' => ['timeout' => 3]]);
        return @$this->fileGetContents($url, false, $ctx) ?: '';
    }

    /**
     * Get geolocation data from a given URL
     *
     * @param string $url
     * @return array
     */
    protected function fetchJson(string $url): array
    {
        $json = $this->fetchText($url);
        return json_decode($json, true) ?: [];
    }

    /**
     * Wrapper so we can easily mock file_get_contents
     *
     * @codeCoverageIgnore
     */
    protected function fileGetContents(string $url, $use_include_path = false, $context = null, $offset = 0, $maxlen = null)
    {
        return file_get_contents($url, $use_include_path, $context, $offset, $maxlen);
    }
}
