<?php
/*
 * CCDADocRefGenerationTest.php
 * @package openemr
 * @link      http://www.open-emr.org
 * @author    Stephen Nielson <snielson@discoverandchange.com>
 * @copyright Copyright (c) 2025 Stephen Nielson <snielson@discoverandchange.com>
 * @license   https://github.com/openemr/openemr/blob/master/LICENSE GNU General Public License 3
 */

namespace OpenEMR\Tests\Certification\HIT1\G9_Certification;

use DOMXPath;
use OpenEMR\Tests\Api\ApiTestClient;
use PHPUnit\Framework\TestCase;
use Symfony\Component\HttpFoundation\Response;

class CCDADocRefGenerationTest extends TestCase {

    // due to the code sets used in the CCDA generation we have to use a specific expected output file that's stored
    // in a private location not accessible to the public repo
    const EXPECTED_RESPONSE_CCDA = __DIR__ . '/../../../../../ci/inferno/inferno-files/files/data/ccda/CCDA_Newman_Alice_expected.xml';

    const DEFAULT_OPENEMR_BASE_URL_API = 'http://openemr';
    const PATIENT_ID_PRIMARY = '96506861-511f-4f6d-bc97-b65a78cf1995';
    private static ApiTestClient $testClient;
    private static string $baseUrl;

    public static function setUpBeforeClass(): void
    {
        parent::setUpBeforeClass(); // TODO: Change the autogenerated stub
        $baseUrl = getenv("OPENEMR_BASE_URL_API", true) ?: self::DEFAULT_OPENEMR_BASE_URL_API;
        self::$testClient = new ApiTestClient($baseUrl, false);
        self::$baseUrl = $baseUrl;
        // for now this uses the admin user to authenticate
        // TODO: @adunsulag need to implement this using a test practitioner user so we can test the inferno single patient API from a regular provider
        self::$testClient->setAuthToken(ApiTestClient::OPENEMR_AUTH_ENDPOINT);
    }

    public function testGenerateCCDADocRef(): void
    {
        if (!file_exists(self::EXPECTED_RESPONSE_CCDA)) {
            $this->markTestSkipped('Expected CCDA response file not found. Running test w/o private repo so Skipping test.');
        }
        // TODO: @snielson we would rather use post with parameters but the current implementation uses get parameters for the patient
        $response = self::$testClient->post("/apis/default/fhir/DocumentReference/\$docref?patient=" . self::PATIENT_ID_PRIMARY, []);

        $result = $response->getBody()->getContents();
        $this->assertNotEmpty($result);
        $data = json_decode($result, true);

        $this->assertArrayHasKey('entry', $data, "Expected 'entry' key in the operation response");
        $entry = $data['entry'];
        $this->assertNotEmpty($entry, "Expected at least one entry in the operation bundle response");
        $this->assertArrayHasKey("resource", $entry[0], "Expected 'resource' key in the first entry of the operation response");
        $docReferenceResource = $entry[0]['resource'];
        $this->assertArrayHasKey('content', $docReferenceResource, "Expected 'content' key in the DocumentReference resource");
        $contentArray = $docReferenceResource['content'];
        $this->assertNotEmpty($contentArray, "Expected at least one content item in the DocumentReference resource");
        $this->assertArrayHasKey('attachment', $contentArray[0], "Expected 'attachment' key in the first content item of the DocumentReference resource");
        $attachment = $contentArray[0]['attachment'];
        $this->assertArrayHasKey('url', $attachment, "Expected 'url' key in the attachment of the DocumentReference resource");
        $this->assertStringStartsWith("/fhir/Binary/", $attachment['url'], "Expected attachment URL to start with '/fhir/Binary/'");
        $url = $attachment['url'];

        $ccdaResponse = self::$testClient->get("/apis/default" . $url);
        $this->assertEquals(Response::HTTP_OK, $ccdaResponse->getStatusCode(), "Expected 200 OK response when retrieving CCDA Binary resource");
        $ccdaContent = $ccdaResponse->getBody()->getContents();
        $this->assertStringContainsString('<?xml version="1.0" encoding="UTF-8"?>', $ccdaContent, "Expected CCDA content to contain XML declaration");
        // now we will do a strict comparison check on the ccda generated document to ensure it matches expected output
        // eventually we'd want a deeper validation of the CCDA structure but for now this will do to verify generation and retrieval
        $expectedCcdaContent = file_get_contents(self::EXPECTED_RESPONSE_CCDA);
        $domXMLContent = $this->getDomXml($ccdaContent);
        $domXMLExpectedContent = $this->getDomXml($expectedCcdaContent);
        $domXMLExpectedContent = $this->cleanWhitespaceInTextNodes($domXMLExpectedContent);
        $docGeneratedTime = $this->getDocumentGenerationTime($domXMLContent);
        $docNewTime = $this->getDocumentGenerationTime($domXMLExpectedContent);
        $updatedDomXMLContent = $this->replaceLatestTimeStamp($domXMLContent, $docGeneratedTime, $docNewTime);

        $updatedDomXMLContent = $this->updateRootIds($updatedDomXMLContent, $domXMLExpectedContent);
        // updating the root nodes and timestamps is cleaning up whitespace in the updated DOMDocument
        // these unfortunately are causing issues with the comparison
        $updatedDomXMLContent = $this->cleanWhitespaceInTextNodes($updatedDomXMLContent);
        $this->assertXmlStringEqualsXmlString($domXMLExpectedContent->C14N(), $updatedDomXMLContent->C14N(), "The generated CCDA document does not match the expected content.");
    }

    private function getDomXml(string $xmlContent): \DOMDocument
    {
        $dom = new \DOMDocument();
        $dom->preserveWhiteSpace = false;
        $dom->formatOutput = false;
        if (!$dom->loadXML($xmlContent, LIBXML_NOBLANKS)) {
            throw new \RuntimeException('Invalid XML');
        }
        return $dom;
    }
    private function getDocumentGenerationTime(\DOMDocument $xml) : string {
        $xpath = new \DOMXPath($xml);
        $xpath->registerNamespace('hl7', 'urn:hl7-org:v3');
        $effectiveTime = $xpath->query("//hl7:effectiveTime")->item(0);
        if ($effectiveTime && $effectiveTime->hasAttribute('value')) {
            return $effectiveTime->getAttribute('value');
        } else {
            throw new \RuntimeException('effectiveTime element with value attribute not found in XML');
        }
    }
    private function replaceLatestTimeStamp(\DOMDocument $xml, string $currentTimestamp, string $newTimeStamp) : \DOMDocument {
        $xpath = new \DOMXPath($xml);
        $expr = '//*[@value="' . $currentTimestamp .  '"]';
        $timestampValues = $xpath->query($expr);
        foreach ($timestampValues as $timestamp) {
            $timestamp->setAttribute('value', $newTimeStamp);
        }
        return $xml;
    }

    private function updateRootIds(\DOMDocument $updatedDomXMLContent, \DOMDocument $domXMLExpectedContent)
    {
        // TODO: We need to figure out why we are generating unique identifiers for these fields that change on
        // each document generation instead of using something the db uuid fields.
        // I don't know if that is required per spec, or if that's a bug in the CCDA generation code.
        $xpath = new DOMXPath($updatedDomXMLContent);
        $xpath->registerNamespace('hl7', 'urn:hl7-org:v3');
        $xpathExpected = new DOMXPath($domXMLExpectedContent);
        $xpathExpected->registerNamespace('hl7', 'urn:hl7-org:v3');
        // need to swap out the 76691-5 Gender Identity root id (only 1)
        // need to swap out the 46098-0 Sex root id (only 1)
        // need to swap out the 76690-7 Sexual Orientation (only 1)
        // need to swap out the 86744-0 Care Team (only 1)
        // need to swap out the 85847-2 Patient Care Team Information section...
        $this->replaceRootIdForXpathQuery("//hl7:observation/hl7:code[@code='76691-5']", $xpath, $xpathExpected);
        $this->replaceRootIdForXpathQuery("//hl7:observation/hl7:code[@code='46098-0']", $xpath, $xpathExpected);
        $this->replaceRootIdForXpathQuery("//hl7:observation/hl7:code[@code='76690-7']", $xpath, $xpathExpected);
        $this->replaceRootIdForXpathQuery("//hl7:section/hl7:entry/hl7:organizer/hl7:code[@code='86744-0']", $xpath, $xpathExpected);
        $this->replaceRootIdForXpathQuery("//hl7:component/hl7:act/hl7:code[@code='85847-2']", $xpath, $xpathExpected);
        return $updatedDomXMLContent;
    }
    private function replaceRootIdForXpathQuery(string $query, DOMXPath $path, DOMXPath $expectedXpath): void {
        $currentList1 = $path->query($query);
        $expectedList2 = $expectedXpath->query($query);
        $this->replaceRootIdForNodes($path, $expectedXpath, $currentList1, $expectedList2);
    }
    private function replaceRootIdForNodes(DOMXPath $path, DOMXPath $expectedXpath, \DOMNodeList $currentList1, \DOMNodeList $expectedList2): void {
        $count = $currentList1->count();
        if ($currentList1->count() != $expectedList2->count()) {
            throw new \RuntimeException('Node lists have different counts');
        }
        for ($i = 0; $i < $count; $i++) {
            $currentNode = $currentList1->item($i)->parentElement;
            $expectedNode = $expectedList2->item($i)->parentElement;
            $currentNodeId = $path->query(".//hl7:id", $currentNode)->item(0);
            $expectedNodeId = $expectedXpath->query(".//hl7:id", $expectedNode)->item(0);
            $expectedRootId = $expectedNodeId->getAttribute('root');
            $currentNodeId->setAttribute('root', $expectedRootId);
        }
    }

    private function cleanWhitespaceInTextNodes(\DOMDocument $updatedDomXMLContent)
    {
        $xp = new DOMXPath($updatedDomXMLContent);
        $xp->registerNamespace('hl7', 'urn:hl7-org:v3');
        $xp->registerNamespace('xhtml', 'http://www.w3.org/1999/xhtml');
        $nodes = $xp->query('//hl7:text//text() | //xhtml:td//text() | //hl7:value//text()');

        foreach ($nodes as $text) {
            $text->nodeValue = trim((string) preg_replace('/\s+/u', ' ', (string) $text->nodeValue));
        }
        return $updatedDomXMLContent;
    }
}
