<?php

/*
 * FhirObservationQuestionnaireItemServiceTest.php - Test class for FhirObservationQuestionnaireItemService
 * @package openemr
 * @link      http://www.open-emr.org
 * @author    Claude.ai generated code: Public Domain generated on 2025-08-25
 * @author    Stephen Nielson <snielson@discoverandchange.com>
 * @copyright Initial copyright is public domain as generated by Claude.ai on 2025-08-25
 * @license   https://github.com/openemr/openemr/blob/master/LICENSE GNU General Public License 3
 */

namespace OpenEMR\Tests\Services\FHIR\Observation;

use OpenEMR\FHIR\R4\FHIRDomainResource\FHIRObservation;
use OpenEMR\FHIR\R4\FHIRDomainResource\FHIRProvenance;
use OpenEMR\FHIR\R4\FHIRElement\FHIRCanonical;
use OpenEMR\FHIR\R4\FHIRElement\FHIRCodeableConcept;
use OpenEMR\FHIR\R4\FHIRElement\FHIRDateTime;
use OpenEMR\FHIR\R4\FHIRElement\FHIRMeta;
use OpenEMR\FHIR\R4\FHIRElement\FHIRPeriod;
use OpenEMR\FHIR\R4\FHIRElement\FHIRReference;
use OpenEMR\Services\CodeTypesService;
use OpenEMR\Services\FHIR\FhirCodeSystemConstants;
use OpenEMR\Services\FHIR\FhirProvenanceService;
use OpenEMR\Services\FHIR\Observation\FhirObservationObservationFormService;
use OpenEMR\Services\FHIR\UtilsService;
use OpenEMR\Services\Search\SearchFieldType;
use PHPUnit\Framework\TestCase;

class FhirObservationObservationFormServiceTest extends TestCase
{
    private FhirObservationObservationFormService $fhirService;
    protected function setUp(): void
    {
        parent::setUp(); // TODO: Change the autogenerated stub
        $this->fhirService = new FhirObservationObservationFormService();
    }

    /**
     * Gets a minimal required record for testing mandatory fields only
     */
    private function getMinimalObservationRecord()
    {
        return [
            'uuid' => 'observation-minimal-123',
            'date' => '2024-01-15 10:30:00',
            'last_updated_time' => '2024-01-15 10:30:00',
            'status' => 'final',
            'puuid' => 'patient-uuid-123',
            'code' => 'LOINC:72133-2',
            'code_description' => 'Alcohol screening',
            'category_code' => 'survey',
            'user_uuid' => 'user-uuid-123',
            'ob_value' => '2',
            'value_type' => 'Quantity'
        ];
    }

    private function getDefaultObservationRecord()
    {
        return [
            'uuid' => 'observation-123',
            'last_updated_time' => date('Y-m-d H:i:s'),
            'date' => date('Y-m-d H:i:s'),
            'user' => 'admin'
            ,'user_uuid' => 'user-123'
            ,'puuid' => 'patient-123'
            ,'code' => 'ICD10:A18.01'
            ,'code_description' => 'Tuberculosis of spine' // our code descriptions have newlines in them... we should look at cleaning that up
            // TODO: @adunsulag need lots of different tests to test the value property
            ,'ob_value' => '120'
            ,'ob_unit' => 'lb'
            ,'sub_observations' => [
                [
                    'uuid' => 'observation-124'
                ]
                ,[
                    'uuid' => 'observation-125'
                ]
            ]
            // the order of derivedFrom responses will be first QuestionnaireResponse if any, followed by Observation if any
            // if questionnaire_response_uuid, will have a derivedFrom QuestionnaireResponse resource
            ,'questionnaire_response_uuid' => 'questionnaire-response-123'
            // if parent_observation_uuid is not null, will have a derivedFrom Observation resource
            ,'parent_observation_uuid' => 'observation-parent-123'
            ,'encounter_uuid' => 'encounter-123'
            ,'note' => 'This is a test note'
        ];
    }

    public function testParseOpenEMRRecordScreeningAssessmentObservationDataAbsentReasonForValue(): void
    {
        $record = $this->getDefaultObservationRecord();
        $record['ob_value'] = null;
        unset($record['sub_observations']); // No hasMember for this test to pass us-core-2 constraint
        $observation = $this->fhirService->parseOpenEMRRecord($record);

        // now we are going to verify the observation
        $this->assertInstanceOf(FHIRObservation::class, $observation, "Observation should have been created");
        $this->assertEmpty($observation->getValueQuantity());
        $this->assertEmpty($observation->getValueCodeableConcept());
        $this->assertEmpty($observation->getValueString());

        $this->assertNotEmpty($observation->getDataAbsentReason(), "Data Absent Reason should have been created");
    }

    public function testParseOpenEMRRecordScreeningAssessmentObservation(): void
    {
        $mockCodeTypeServices = $this->createMock(CodeTypesService::class);
        $mockCodeTypeServices->expects($this->once())
            ->method("lookup_code_description")
            ->willReturn("Tuberculosis of spine");
        $mockCodeTypeServices->expects($this->once())
            ->method('parseCode')
            ->willReturn([
                'code_type' => "ICD10",
                'code' => 'A18.01',
                'display' => 'Tuberculosis of spine'
            ]);
        $mockCodeTypeServices->expects($this->once())
            ->method('getSystemForCodeType')
            ->with('ICD10')
            ->willReturn(FhirCodeSystemConstants::HL7_ICD10);

        $this->fhirService->setCodeTypesService($mockCodeTypeServices);
        $record = $this->getDefaultObservationRecord();
        $observation = $this->fhirService->parseOpenEMRRecord($record);

        // now we are going to verify the observation
        $this->assertInstanceOf(FHIRObservation::class, $observation, "Observation should have been created");
        $this->assertEquals('1', $observation->getMeta()->getVersionId());

        // Test required category with survey slice (mustSupport, min 1)
        $categories = $observation->getCategory();
        $this->assertGreaterThanOrEqual(1, count($categories));

        $surveyCategory = $categories[0];
        $this->assertInstanceOf(FHIRCodeableConcept::class, $surveyCategory);
        $coding = $surveyCategory->getCoding()[0];
        $this->assertEquals('http://terminology.hl7.org/CodeSystem/observation-category', $coding->getSystem());
        $this->assertEquals('survey', $coding->getCode());

        // Test additional screening-assessment category if present
        if (isset($record['screening_category_code'])) {
            $this->assertCount(2, $categories);
            $screeningCategory = $categories[1];
            $screeningCoding = $screeningCategory->getCoding()[0];
            $this->assertEquals($record['screening_category_system'], $screeningCoding->getSystem());
            $this->assertEquals($record['screening_category_code'], $screeningCoding->getCode());
        }

        $this->assertEquals($record['uuid'], $observation->getId()->getValue(), "Uuid should be populated");

        // Test required effective[x] field (mustSupport, dateTime must be supported)
        $this->assertInstanceOf(FHIRDateTime::class, $observation->getEffectiveDateTime());
        $expectedDate = UtilsService::getLocalDateAsUTC($record['date']);
        $this->assertEquals($expectedDate, $observation->getEffectiveDateTime()->getValue());

        // Test effectivePeriod when date_end is present
        if (isset($record['date_end'])) {
            $this->assertInstanceOf(FHIRPeriod::class, $observation->getEffectivePeriod());
            $expectedEndDate = UtilsService::getLocalDateAsUTC($record['date_end']);
            $this->assertEquals($expectedEndDate, $observation->getEffectivePeriod()->getEnd()->getValue());
        }

        $this->assertNotEmpty($observation->getCode(), "Code property should be populated");
        $this->assertCount(1, $observation->getCode()->getCoding(), "Coding should have just a single code in  it");
        $this->assertNotEmpty($observation->getCode()->getCoding()[0]->getSystem());
        $this->assertEquals(FhirCodeSystemConstants::HL7_ICD10, $observation->getCode()->getCoding()[0]->getSystem(), "Code.coding[0].system should have been set");
        $this->assertNotEmpty($observation->getCode()->getCoding()[0]->getCode(), "observation.code.coding[0].code should not be empty");
        $this->assertNotEmpty($observation->getCode()->getCoding()[0]->getDisplay(), "observation.code.coding[0].display should not be empty");
        $this->assertEquals(substr((string) $record['code'], 6), $observation->getCode()->getCoding()[0]->getCode());
        $this->assertEquals($record['code_description'], $observation->getCode()->getCoding()[0]->getDisplay());

        // Test required performer field (mustSupport)
        $this->assertNotEmpty($observation->getPerformer());
        $this->assertEquals('Practitioner/' . $record['user_uuid'], $observation->getPerformer()[0]->getReference());

        // Test required value[x] field (mustSupport, constraint us-core-2)
        $this->assertNotEmpty($observation->getValueQuantity());
        $this->assertEquals($record['ob_value'], $observation->getValueQuantity()->getValue());

        // Test UCUM constraint (us-core-3) - should use UCUM units when system is specified
        if ($record['ob_unit'] !== '{score}') {
            $this->assertEquals('http://unitsofmeasure.org', $observation->getValueQuantity()->getSystem()->getValue());
        }

        $this->assertNotEmpty($observation->getValueQuantity(), "Quantity should be populated");
        $this->assertEquals($record['ob_value'], $observation->getValueQuantity()->getValue(), "Quantity value should be populated");
        $this->assertEquals($record['ob_unit'], $observation->getValueQuantity()->getUnit(), "Quantity unit should be populated");

        // Test required hasMember field (mustSupport) for panel observations
        if (!empty($record['sub_observations'])) {
            $this->assertCount(count($record['sub_observations']), $observation->getHasMember());
            $this->assertEquals(
                'Observation/' . $record['sub_observations'][0]['uuid'],
                $observation->getHasMember()[0]->getReference()->getValue()
            );
        }

        // Test required derivedFrom field (mustSupport)
        $derivedFromRefs = $observation->getDerivedFrom();
        $expectedDerivedFromCount = 0;
        // TODO: @adunsulag re-enable when questionnaire responses are supported validating via inferno
//        if (!empty($record['questionnaire_response_uuid'])) {
//            $expectedDerivedFromCount++;
//        }
        if (!empty($record['parent_observation_uuid'])) {
            $expectedDerivedFromCount++;
        }

        $this->assertCount($expectedDerivedFromCount, $derivedFromRefs);

//        if (!empty($record['questionnaire_response_uuid'])) {
//            $this->assertEquals(
//                'QuestionnaireResponse/' . $record['questionnaire_response_uuid'],
//                $derivedFromRefs[0]->getReference()->getValue()
//            );
//        }
        if (!empty($record['parent_observation_uuid'])) {
            $this->assertEquals(
                'Observation/' . $record['parent_observation_uuid'],
                $derivedFromRefs[$expectedDerivedFromCount - 1]->getReference()->getValue()
            );
        }
    }

    /**
     * Test parsing minimal observation record with only required fields
     */
    public function testParseOpenEMRRecordMinimal(): void
    {
        $record = $this->getMinimalObservationRecord();
        $observation = $this->fhirService->parseOpenEMRRecord($record);

        // Verify all required fields are present
        $this->assertInstanceOf(FHIRObservation::class, $observation);
        $this->assertNotNull($observation->getStatus());
        $this->assertNotEmpty($observation->getCategory());
        $this->assertNotNull($observation->getCode());
        $this->assertNotNull($observation->getSubject());
        $this->assertNotNull($observation->getEffectiveDateTime());
        $this->assertNotEmpty($observation->getPerformer());
        $this->assertNotNull($observation->getValueQuantity());
    }

    /**
     * Test observation with dataAbsentReason instead of value (us-core-2 constraint)
     */
    public function testParseOpenEMRRecordDataAbsentReasonForValue(): void
    {
        $record = $this->getDefaultObservationRecord();
        $record['ob_value'] = null;
        unset($record['sub_observations']); // No hasMember for this test to pass us-core-2 constraint

        $observation = $this->fhirService->parseOpenEMRRecord($record);

        // Verify value fields are empty
        $this->assertEmpty($observation->getValueQuantity());
        $this->assertEmpty($observation->getValueCodeableConcept());
        $this->assertEmpty($observation->getValueString());

        // Verify dataAbsentReason is populated (mustSupport)
        $this->assertNotEmpty($observation->getDataAbsentReason());
        $dataAbsentCoding = $observation->getDataAbsentReason()->getCoding()[0];
        $this->assertEquals(FhirCodeSystemConstants::DATA_ABSENT_REASON_CODE_SYSTEM, $dataAbsentCoding->getSystem());
        $this->assertEquals('unknown', $dataAbsentCoding->getCode());
    }

    /**
     * Test observation with string value type
     */
    public function testParseOpenEMRRecordStringValue(): void
    {
        $record = $this->getDefaultObservationRecord();
        $record['ob_value'] = 'Never';
        unset($record['ob_unit']);

        $observation = $this->fhirService->parseOpenEMRRecord($record);

        $this->assertNotEmpty($observation->getValueString());
        $this->assertEquals($record['ob_value'], $observation->getValueString()->getValue());
        $this->assertEmpty($observation->getValueQuantity());
    }

    /**
     * Test observation with CodeableConcept value type
     */
    public function testParseOpenEMRRecordCodeableConceptValue(): void
    {
        $record = $this->getDefaultObservationRecord();
        $codeSystem = 'LOINC';
        $record['ob_value'] = "$codeSystem:LA6111-4";
        $record['ob_value_code_description'] = 'Tuberculosis screening';

        $observation = $this->fhirService->parseOpenEMRRecord($record);

        $this->assertNotEmpty($observation->getValueCodeableConcept());
        $valueCoding = $observation->getValueCodeableConcept()->getCoding()[0];
        $this->assertEquals(FhirCodeSystemConstants::LOINC, $valueCoding->getSystem()->getValue());
        $this->assertEquals(substr((string) $record['ob_value'], strlen($codeSystem) + 1), $valueCoding->getCode()->getValue());
    }

    /**
     * Test observation with encounter reference
     */
    public function testParseOpenEMRRecordWithEncounter(): void
    {
        $record = $this->getDefaultObservationRecord();

        $observation = $this->fhirService->parseOpenEMRRecord($record);

        $this->assertNotEmpty($observation->getEncounter());
        $this->assertEquals('Encounter/' . $record['encounter_uuid'], $observation->getEncounter()->getReference()->getValue());
    }

    /**
     * Test observation with note/annotation
     */
    public function testParseOpenEMRRecordWithNote(): void
    {
        $record = $this->getDefaultObservationRecord();

        $observation = $this->fhirService->parseOpenEMRRecord($record);

        $this->assertNotEmpty($observation->getNote());
        $this->assertEquals($record['note'], $observation->getNote()[0]->getText());
    }

    /**
     * Test observation with interpretation - TODO: @adunsulag intrepretaion's are not currently collected... may need to add to form, or remove this field
     */
//    public function testParseOpenEMRRecordWithInterpretation()
//    {
//        $record = $this->getDefaultObservationRecord();
//        $record['interpretation'] = 'H';
//        $record['interpretation_display'] = 'High';
//
//        $observation = $this->fhirService->parseOpenEMRRecord($record);
//
//        $this->assertNotEmpty($observation->getInterpretation());
//        $interpretationCoding = $observation->getInterpretation()[0]->getCoding()[0];
//        $this->assertEquals($record['interpretation_system'], $interpretationCoding->getSystem()->getValue());
//        $this->assertEquals($record['interpretation'], $interpretationCoding->getCode()->getValue());
//    }

    /**
     * Test observation for panel/group with hasMember but no value (us-core-2 constraint)
     */
    public function testParseOpenEMRRecordPanelObservation(): void
    {
        $record = $this->getDefaultObservationRecord();
        unset($record['ob_value'], $record['ob_unit']);
        $record['data_absent_reason'] = null; // Panel has members, no individual value needed
        $observation = $this->fhirService->parseOpenEMRRecord($record);

        // Panel should have hasMember but no value
        $this->assertNotEmpty($observation->getHasMember());
        $this->assertEmpty($observation->getValueQuantity());
        $this->assertEmpty($observation->getValueString());
        $this->assertEmpty($observation->getValueCodeableConcept());
        $this->assertEmpty($observation->getDataAbsentReason()); // Not needed when hasMember present
    }

    /**
     * Test datetime constraint (us-core-1) - must be at least to day precision
     */
    public function testParseOpenEMRRecordDateTimePrecision(): void
    {
        $record = $this->getDefaultObservationRecord();
        $record['date'] = '2024-01-15'; // Date only, should be valid

        $observation = $this->fhirService->parseOpenEMRRecord($record);

        $dateTime = $observation->getEffectiveDateTime();
        $this->assertGreaterThanOrEqual(10, strlen($dateTime)); // At least YYYY-MM-DD format
    }

    /**
     * Test US Core profile metadata
     */
    public function testParseOpenEMRRecordUSCoreProfile(): void
    {
        $record = $this->getDefaultObservationRecord();
        $observation = $this->fhirService->parseOpenEMRRecord($record);

        // Verify US Core Screening Assessment profile is set in meta
        $profiles = $observation->getMeta()->getProfile();
        $profileValues = array_map(fn($uri): string => $uri->getValue(), $profiles);
        $this->assertContains(FhirObservationObservationFormService::USCGI_PROFILE_URI, $profileValues);
    }

    /**
     * Test different performer types
     */
    public function testParseOpenEMRRecordDifferentPerformerTypes(): void
    {
        $performerTypes = [
            'Patient' => 'patient-uuid-456',
            'Organization' => 'organization-uuid-789'
        ];

        foreach ($performerTypes as $type => $uuid) {
            $record = $this->getDefaultObservationRecord();
            $record['performer_type'] = $type;
            $record['performer_uuid'] = $uuid;

            $observation = $this->fhirService->parseOpenEMRRecord($record);

            $this->assertNotEmpty($observation->getPerformer());
            $expectedRef = $type . '/' . $uuid;
            $this->assertEquals($expectedRef, $observation->getPerformer()[0]->getReference());
        }
    }

    /**
     * Test missing status defaults to 'unknown' (required field)
     */
    public function testParseOpenEMRRecordMissingStatusSetToUnknown(): void
    {
        // Test missing status
        $record = $this->getMinimalObservationRecord();
        unset($record['status']);
        $observation = $this->fhirService->parseOpenEMRRecord($record);
        $this->assertEquals('unknown', $observation->getStatus()->getValue(), "Missing status should default to 'unknown'");
    }

    public function testParseOpenEMRRecordInvalidStatus(): void
    {
        $record = $this->getMinimalObservationRecord();
        $record['status'] = 'invalid-status';

        // Should default to 'unknown' for invalid status
        $observation = $this->fhirService->parseOpenEMRRecord($record);
        $this->assertEquals('unknown', $observation->getStatus()->getValue());
    }

    public function testGetPatientContextSearchField(): void
    {
        $fhirSearchDefinition = $this->fhirService->getPatientContextSearchField();
        $this->assertCount(1, $fhirSearchDefinition->getMappedFields());
        $this->assertEquals("puuid", $fhirSearchDefinition->getMappedFields()[0]->getField());
        $this->assertEquals("patient", $fhirSearchDefinition->getName());
        $this->assertEquals(SearchFieldType::REFERENCE, $fhirSearchDefinition->getType());
    }

    public function testSupportsCategory(): void
    {
        // TODO: @adunsulag do we want to hard-code these categories or pull them from a list service?
        $categories = FhirObservationObservationFormService::SUPPORTED_CATEGORIES;
        foreach ($categories as $category) {
            $this->assertTrue($this->fhirService->supportsCategory($category), "service should support the category of " . $category);
        }
    }

    public function testSupportsCode(): void
    {
        $this->markTestIncomplete("Need to implement this test");
    }

    public function testCreateProvenanceResource(): void
    {

        $performer = new FHIRReference();
        $performer->setReference("Practitioner/uuid-123");
        $fhirObservation = new FHIRObservation();
        $fhirObservation->setId("observation-123");
        $meta = new FHIRMeta();
        $meta->setVersionId('1');
        $meta->setLastUpdated(UtilsService::getDateFormattedAsUTC());
        $fhirObservation->setMeta($meta);
        $fhirObservation->addPerformer($performer);
        $provenanceService = $this->createMock(FhirProvenanceService::class);
        $fhirProvenance = new FHIRProvenance();
        $provenanceService->expects($this->any())
            ->method('createProvenanceForDomainResource')
            ->with($fhirObservation, $performer)
            ->willReturn(
                $fhirProvenance
            );
        $this->fhirService->setProvenanceService($provenanceService);
        $provenance = $this->fhirService->createProvenanceResource($fhirObservation);
        $this->assertSame($fhirProvenance, $provenance);
    }
}
