<?php

/*
 * FhirObservationQuestionnaireItemServiceTest.php - Test class for FhirObservationQuestionnaireItemService
 * @package openemr
 * @link      http://www.open-emr.org
 * @author    Claude.ai generated code: Public Domain generated on 2025-08-25
 * @author    Stephen Nielson <snielson@discoverandchange.com>
 * @copyright Initial copyright is public domain as generated by Claude.ai on 2025-08-25
 * @license   https://github.com/openemr/openemr/blob/master/LICENSE GNU General Public License 3
 */

namespace OpenEMR\Tests\Services\FHIR\Observation;

use OpenEMR\FHIR\R4\FHIRDomainResource\FHIRObservation;
use OpenEMR\FHIR\R4\FHIRDomainResource\FHIRProvenance;
use OpenEMR\FHIR\R4\FHIRElement\FHIRCodeableConcept;
use OpenEMR\FHIR\R4\FHIRElement\FHIRDateTime;
use OpenEMR\FHIR\R4\FHIRElement\FHIRMeta;
use OpenEMR\FHIR\R4\FHIRElement\FHIRPeriod;
use OpenEMR\FHIR\R4\FHIRElement\FHIRReference;
use OpenEMR\Services\CodeTypesService;
use OpenEMR\Services\FHIR\FhirCodeSystemConstants;
use OpenEMR\Services\FHIR\FhirProvenanceService;
use OpenEMR\Services\FHIR\Observation\FhirObservationObservationFormService;
use OpenEMR\Services\FHIR\Observation\FhirObservationQuestionnaireItemService;
use OpenEMR\Services\FHIR\UtilsService;
use OpenEMR\Services\Search\SearchFieldType;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;

class FhirObservationQuestionnaireItemServiceTest extends TestCase
{
    private FhirObservationQuestionnaireItemService $fhirService;
    private MockObject&CodeTypesService $codeTypesService;
    protected function setUp(): void
    {
        parent::setUp(); // TODO: Change the autogenerated stub
        $this->fhirService = new FhirObservationQuestionnaireItemService();
        $this->codeTypesService = $this->createPartialMock(CodeTypesService::class, [
            'lookup_code_description'
        ]);
        $this->fhirService->setCodeTypesService($this->codeTypesService);
    }

    /**
     * Gets a minimal required record for testing mandatory fields only
     */
    private function getMinimalObservationRecord()
    {
        return [
            'uuid' => 'questionnaire-item-minimal-123',
            'date' => '2024-01-15 10:30:00',
            'status' => 'final',
            'puuid' => 'patient-uuid-123',
            'code' => 'LOINC:72133-2',
            'code_description' => 'Alcohol screening',
            'category_code' => 'survey',
            'user_uuid' => 'user-uuid-123',
            'value' => '2',
            'value_unit' => '{score}'
        ];
    }

    private function getDefaultObservationRecord()
    {
        return [
            'uuid' => 'questionnaire-item-123'
            ,'date' => date('Y-m-d H:i:s')
            ,'user' => 'admin'
            ,'user_uuid' => 'user-123'
            ,'puuid' => 'patient-123'
            ,'code' => 'LOINC:72109-2'
            ,'code_description' => 'Alcohol Use Disorder Identification Test - Consumption [AUDIT-C]' // our code descriptions have newlines in them... we should look at cleaning that up
            ,'value' => '120'
            ,'value_unit' => 'lb'
            ,'sub_observations' => [
                [
                    'uuid' => 'questionnaire-item-1'
                    ,'code' => 'LOINC:68519-0'
                    ,'value' => 'Never'
                    ,'description' => 'How often do you have a drink containing alcohol?'
                ]
                ,[
                    'uuid' => 'questionnaire-item-2'
                    ,'code' => 'LOINC:68519-8'
                    ,'value' => '2'
                    ,'value_unit' => '{#}/d'
                    ,'description' => 'How many standard drinks containing alcohol do you have on a typical day?'
                ]
                ,[
                    'uuid' => 'questionnaire-item-3'
                    ,'code' => 'LOINC:68520-6'
                    ,'value' => 'Monthly or less'
                    ,'description' => 'How often do you have six or more drinks on one occasion?'
                ]
                ,[
                    'uuid' => 'questionnaire-item-4'
                    ,'code' => 'LOINC:75626-2'
                    ,'value' => '2'
                    ,'value_unit' => '{score}'
                    ,'description' => 'AUDIT-C Total Score'
                ]
            ]
            // the order of derivedFrom responses will be first QuestionnaireResponse if any, followed by Observation if any
            // if questionnaire_response_uuid, will have a derivedFrom QuestionnaireResponse resource
            ,'questionnaire_response_uuid' => 'questionnaire-response-123'
            // if parent-uuid is not null, will have a derivedFrom Observation resource
            ,'parent_uuid' => null
            ,'encounter_uuid' => 'encounter-123'
            ,'note' => 'This is a test note'
        ];
    }

    public function testParseOpenEMRRecordScreeningAssessmentObservationDataAbsentReasonForValue(): void
    {
        $record = $this->getDefaultObservationRecord();
        $record['value'] = null;
        unset($record['sub_observations']); // No hasMember for this test to pass us-core-2 constraint
        $observation = $this->fhirService->parseOpenEMRRecord($record);

        // now we are going to verify the observation
        $this->assertInstanceOf(FHIRObservation::class, $observation, "Observation should have been created");
        $this->assertEmpty($observation->getValueQuantity());
        $this->assertEmpty($observation->getValueCodeableConcept());
        $this->assertEmpty($observation->getValueString());

        $this->assertNotEmpty($observation->getDataAbsentReason(), "Data Absent Reason should have been created");
    }

    public function testParseOpenEMRRecordScreeningAssessmentObservation(): void
    {
        $record = $this->getDefaultObservationRecord();
        $this->codeTypesService->method('lookup_code_description')
            ->with($record['code'])
            ->willReturn($record['code_description']);
        $observation = $this->fhirService->parseOpenEMRRecord($record);

        // now we are going to verify the observation
        $this->assertInstanceOf(FHIRObservation::class, $observation, "Observation should have been created");
        $this->assertEquals('1', $observation->getMeta()->getVersionId());

        // Test required category with survey slice (mustSupport, min 1)
        $categories = $observation->getCategory();
        $this->assertGreaterThanOrEqual(1, count($categories));

        $surveyCategory = $categories[0];
        $this->assertInstanceOf(FHIRCodeableConcept::class, $surveyCategory);
        $coding = $surveyCategory->getCoding()[0];
        $this->assertEquals('http://terminology.hl7.org/CodeSystem/observation-category', $coding->getSystem());
        $this->assertEquals('survey', $coding->getCode());

        // Test additional screening-assessment category if present
        if (isset($record['screening_category_code'])) {
            $this->assertCount(2, $categories);
            $screeningCategory = $categories[1];
            $screeningCoding = $screeningCategory->getCoding()[0];
            $this->assertEquals($record['screening_category_system'], $screeningCoding->getSystem());
            $this->assertEquals($record['screening_category_code'], $screeningCoding->getCode());
        }

        $this->assertEquals($record['uuid'], $observation->getId()->getValue(), "Uuid should be populated");

        // Test required effective[x] field (mustSupport, dateTime must be supported)
        $this->assertInstanceOf(FHIRDateTime::class, $observation->getEffectiveDateTime());
        $expectedDate = UtilsService::getLocalDateAsUTC($record['date']);
        $this->assertEquals($expectedDate, $observation->getEffectiveDateTime()->getValue());

        // Test effectivePeriod when date_end is present
        if (isset($record['date_end'])) {
            $this->assertInstanceOf(FHIRPeriod::class, $observation->getEffectivePeriod());
            $expectedEndDate = UtilsService::getLocalDateAsUTC($record['date_end']);
            $this->assertEquals($expectedEndDate, $observation->getEffectivePeriod()->getEnd()->getValue());
        }

        $this->assertNotEmpty($observation->getCode(), "Code property should be populated");
        $this->assertCount(1, $observation->getCode()->getCoding(), "Coding should have just a single code in  it");
        $this->assertNotEmpty($observation->getCode()->getCoding()[0]->getSystem());
        $this->assertEquals(FhirCodeSystemConstants::LOINC, $observation->getCode()->getCoding()[0]->getSystem(), "Code.coding[0].system should have been set");
        $this->assertNotEmpty($observation->getCode()->getCoding()[0]->getCode(), "observation.code.coding[0].code should not be empty");
        $this->assertNotEmpty($observation->getCode()->getCoding()[0]->getDisplay(), "observation.code.coding[0].display should not be empty");
        $this->assertEquals(substr((string) $record['code'], 6), $observation->getCode()->getCoding()[0]->getCode());
        $this->assertEquals($record['code_description'], $observation->getCode()->getCoding()[0]->getDisplay());

        // Test required performer field (mustSupport)
        $this->assertNotEmpty($observation->getPerformer());
        $this->assertNotEmpty($observation->getPerformer()[0]->getReference());
        $this->assertEquals('Practitioner/' . $record['user_uuid'], $observation->getPerformer()[0]->getReference());

        // Test required value[x] field (mustSupport, constraint us-core-2)
        $this->assertNotEmpty($observation->getValueQuantity());
        $this->assertEquals($record['value'], $observation->getValueQuantity()->getValue());

        // Test UCUM constraint (us-core-3) - should use UCUM units when system is specified
        if ($record['value_unit'] !== '{score}') {
            $this->assertEquals('http://unitsofmeasure.org', $observation->getValueQuantity()->getSystem()->getValue());
        }

        $this->assertNotEmpty($observation->getValueQuantity(), "Quantity should be populated");
        $this->assertEquals($record['value'], $observation->getValueQuantity()->getValue(), "Quantity value should be populated");
        $this->assertEquals($record['value_unit'], $observation->getValueQuantity()->getUnit(), "Quantity unit should be populated");

        // Test required hasMember field (mustSupport) for panel observations
        if (!empty($record['sub_observations'])) {
            $this->assertCount(count($record['sub_observations']), $observation->getHasMember());
            $this->assertEquals(
                'Observation/' . $record['sub_observations'][0]['uuid'],
                $observation->getHasMember()[0]->getReference()->getValue()
            );
        }
    }

    public function testParseOpenEMRRecordScreeningAssessmentObservationDerivedFrom(): void
    {
        $this->markTestSkipped("Skipping until we can figure out why inferno validator is failing on this");
//        $record = $this->getDefaultObservationRecord();
//        $record['questionnaire_response_uuid'] = 'questionnaire-response-124';
//        $record['parent_observation_uuid'] = 'parent-observation-124';
//
//        $observation = $this->fhirService->parseOpenEMRRecord($record);
//        // Test required derivedFrom field (mustSupport)
//        $derivedFromRefs = $observation->getDerivedFrom();
//
//        // TODO: @adunsulag when we can figure out why inferno validator is failing on this, we can re-enable the test
//        $this->assertCount(1, $derivedFromRefs);
//
//        $this->assertEquals(
//            'QuestionnaireResponse/' . $record['questionnaire_response_uuid'],
//            $derivedFromRefs[0]->getReference()->getValue()
//        );
//        $this->assertEquals(
//            'Observation/' . $record['parent_observation_uuid'],
//            $derivedFromRefs[1]->getReference()->getValue()
//        );
    }


    /**
     * Test parsing minimal observation record with only required fields
     */
    public function testParseOpenEMRRecordMinimal(): void
    {
        $record = $this->getMinimalObservationRecord();
        $observation = $this->fhirService->parseOpenEMRRecord($record);

        // Verify all required fields are present
        $this->assertInstanceOf(FHIRObservation::class, $observation);
        $this->assertNotNull($observation->getStatus());
        $this->assertNotEmpty($observation->getCategory());
        $this->assertNotNull($observation->getCode());
        $this->assertNotNull($observation->getSubject());
        $this->assertNotNull($observation->getEffectiveDateTime());
        $this->assertNotEmpty($observation->getPerformer());
        $this->assertNotNull($observation->getValueQuantity());
    }

    /**
     * Test observation with dataAbsentReason instead of value (us-core-2 constraint)
     */
    public function testParseOpenEMRRecordDataAbsentReasonForValue(): void
    {
        $record = $this->getDefaultObservationRecord();
        $record['value'] = null;
        unset($record['sub_observations']); // No hasMember for this test to pass us-core-2 constraint

        $observation = $this->fhirService->parseOpenEMRRecord($record);

        // Verify value fields are empty
        $this->assertEmpty($observation->getValueQuantity());
        $this->assertEmpty($observation->getValueCodeableConcept());
        $this->assertEmpty($observation->getValueString());

        // Verify dataAbsentReason is populated (mustSupport)
        $this->assertNotEmpty($observation->getDataAbsentReason());
        $dataAbsentCoding = $observation->getDataAbsentReason()->getCoding()[0];
        $this->assertEquals(FhirCodeSystemConstants::DATA_ABSENT_REASON_CODE_SYSTEM, $dataAbsentCoding->getSystem());
        $this->assertEquals('unknown', $dataAbsentCoding->getCode());
    }

    /**
     * Test observation with string value type
     */
    public function testParseOpenEMRRecordStringValue(): void
    {
        $record = $this->getDefaultObservationRecord();
        $record['value'] = 'Never';
        unset($record['value_unit']);

        $observation = $this->fhirService->parseOpenEMRRecord($record);

        $this->assertNotEmpty($observation->getValueString());
        $this->assertEquals($record['value'], $observation->getValueString()->getValue());
        $this->assertEmpty($observation->getValueQuantity());
    }

    /**
     * Test observation with CodeableConcept value type
     */
    public function testParseOpenEMRRecordCodeableConceptValue(): void
    {
        $record = $this->getDefaultObservationRecord();
        $codeSystem = 'LOINC';
        $record['value'] = "$codeSystem:LA6111-4";
        $record['value_code_description'] = 'Never smoked';
        $this->codeTypesService->method('lookup_code_description')
            ->willReturnMap([
                [$record['code'], $record['code_description']]
                ,[$record['value'], $record['value_code_description']]
            ]);

        $observation = $this->fhirService->parseOpenEMRRecord($record);

        $this->assertNotEmpty($observation->getValueCodeableConcept());
        $valueCoding = $observation->getValueCodeableConcept()->getCoding()[0];
        $this->assertEquals(FhirCodeSystemConstants::LOINC, $valueCoding->getSystem()->getValue());
        $this->assertEquals(substr((string) $record['value'], strlen($codeSystem) + 1), $valueCoding->getCode()->getValue());
    }

    /**
     * Test observation with encounter reference
     */
    public function testParseOpenEMRRecordWithEncounter(): void
    {
        $record = $this->getDefaultObservationRecord();

        $observation = $this->fhirService->parseOpenEMRRecord($record);

        $this->assertNotEmpty($observation->getEncounter());
        $this->assertEquals('Encounter/' . $record['encounter_uuid'], $observation->getEncounter()->getReference()->getValue());
    }

    /**
     * Test observation with note/annotation
     */
    public function testParseOpenEMRRecordWithNote(): void
    {
        $record = $this->getDefaultObservationRecord();

        $observation = $this->fhirService->parseOpenEMRRecord($record);

        $this->assertNotEmpty($observation->getNote());
        $this->assertEquals($record['note'], $observation->getNote()[0]->getText());
    }

    /**
     * Test observation for panel/group with hasMember but no value (us-core-2 constraint)
     */
    public function testParseOpenEMRRecordPanelObservation(): void
    {
        $record = $this->getDefaultObservationRecord();
        unset($record['value'], $record['value_unit']);
        $record['data_absent_reason'] = null; // Panel has members, no individual value needed

        $observation = $this->fhirService->parseOpenEMRRecord($record);

        // Panel should have hasMember but no value
        $this->assertNotEmpty($observation->getHasMember());
        $this->assertEmpty($observation->getValueQuantity());
        $this->assertEmpty($observation->getValueString());
        $this->assertEmpty($observation->getValueCodeableConcept());
        $this->assertEmpty($observation->getDataAbsentReason()); // Not needed when hasMember present
    }

    /**
     * Test datetime constraint (us-core-1) - must be at least to day precision
     */
    public function testParseOpenEMRRecordDateTimePrecision(): void
    {
        $record = $this->getDefaultObservationRecord();
        $record['date'] = '2024-01-15'; // Date only, should be valid

        $observation = $this->fhirService->parseOpenEMRRecord($record);

        $dateTime = $observation->getEffectiveDateTime()->getValue();
        $this->assertGreaterThanOrEqual(10, strlen((string) $dateTime)); // At least YYYY-MM-DD format
    }

    /**
     * Test US Core profile metadata
     */
    public function testParseOpenEMRRecordUSCoreProfile(): void
    {
        $record = $this->getDefaultObservationRecord();
        $observation = $this->fhirService->parseOpenEMRRecord($record);

        // Verify US Core Screening Assessment profile is set in meta
        $profiles = $observation->getMeta()->getProfile();
        $profileValues = array_map(fn($uri) => $uri->getValue(), $profiles);
        $this->assertContains(FhirObservationQuestionnaireItemService::USCGI_PROFILE_URI, $profileValues);
    }

    /**
     * Test different performer types
     */
    public function testParseOpenEMRRecordDifferentPerformerTypes(): void
    {
        $performerTypes = [
            'Patient' => 'patient-uuid-456',
            'Organization' => 'organization-uuid-789'
        ];

        foreach ($performerTypes as $type => $uuid) {
            $record = $this->getDefaultObservationRecord();
            $record['performer_type'] = $type;
            $record['performer_uuid'] = $uuid;

            $observation = $this->fhirService->parseOpenEMRRecord($record);

            $this->assertNotEmpty($observation->getPerformer());
            $expectedRef = $type . '/' . $uuid;
            $this->assertEquals($expectedRef, $observation->getPerformer()[0]->getReference());
        }
    }

    /**
     * Test missing status defaults to 'unknown' (required field)
     */
    public function testParseOpenEMRRecordMissingStatusSetToUnknown(): void
    {
        // Test missing status
        $record = $this->getMinimalObservationRecord();
        unset($record['status']);
        $observation = $this->fhirService->parseOpenEMRRecord($record);
        $this->assertEquals('unknown', $observation->getStatus()->getValue(), "Missing status should default to 'unknown'");
    }

    public function testParseOpenEMRRecordInvalidStatus(): void
    {
        $record = $this->getMinimalObservationRecord();
        $record['status'] = 'invalid-status';

        // Should default to 'unknown' for invalid status
        $observation = $this->fhirService->parseOpenEMRRecord($record);
        $this->assertEquals('unknown', $observation->getStatus()->getValue());
    }

    public function testGetPatientContextSearchField(): void
    {
        $fhirSearchDefinition = $this->fhirService->getPatientContextSearchField();
        $this->assertCount(1, $fhirSearchDefinition->getMappedFields());
        $this->assertEquals("puuid", $fhirSearchDefinition->getMappedFields()[0]->getField());
        $this->assertEquals("patient", $fhirSearchDefinition->getName());
        $this->assertEquals(SearchFieldType::REFERENCE, $fhirSearchDefinition->getType());
    }

    public function testSupportsCategory(): void
    {
        // TODO: @adunsulag do we want to hard-code these categories or pull them from a list service?
        $categories = FhirObservationQuestionnaireItemService::US_CORE_CODESYSTEM_CATEGORY;
        foreach ($categories as $category) {
            $this->assertTrue($this->fhirService->supportsCategory($category), "service should support the category of " . $category);
        }
    }

    public function testSupportsCode(): void
    {
        $this->markTestIncomplete("Need to implement this test");
    }

    public function testCreateProvenanceResource(): void
    {

        $performer = new FHIRReference();
        $performer->setReference("Practitioner/uuid-123");
        $fhirObservation = new FHIRObservation();
        $fhirObservation->setId("observation-123");
        $meta = new FHIRMeta();
        $meta->setVersionId('1');
        $meta->setLastUpdated(UtilsService::getDateFormattedAsUTC());
        $fhirObservation->setMeta($meta);
        $fhirObservation->addPerformer($performer);
        $provenanceService = $this->createMock(FhirProvenanceService::class);
        $fhirProvenance = new FHIRProvenance();
        $provenanceService->expects($this->any())
            ->method('createProvenanceForDomainResource')
            ->with($fhirObservation, $performer)
            ->willReturn(
                $fhirProvenance
            );
        $this->fhirService->setProvenanceService($provenanceService);
        $provenance = $this->fhirService->createProvenanceResource($fhirObservation);
        $this->assertSame($fhirProvenance, $provenance);
        // TODO: @adunsulag assert additional provenance resources here
    }
}
