<?php

/*
 * FhirQuestionnaireFormServiceTest.php
 * @package openemr
 * @link      http://www.open-emr.org
 * @author    Stephen Nielson <snielson@discoverandchange.com>
 * @copyright Copyright (c) 2025 Stephen Nielson <snielson@discoverandchange.com>
 * @copyright Public Domain for sections that were generated with Claude.AI on September 3rd 2025
 * @license   https://github.com/openemr/openemr/blob/master/LICENSE GNU General Public License 3
 */

namespace OpenEMR\Tests\Services\FHIR\QuestionnaireResponse;

use Monolog\Level;
use OpenEMR\Common\Logging\SystemLogger;
use OpenEMR\Common\Uuid\UuidRegistry;
use OpenEMR\FHIR\R4\FHIRDomainResource\FHIRQuestionnaireResponse;
use OpenEMR\Services\FHIR\Questionnaire\FhirQuestionnaireFormService;
use OpenEMR\Services\FHIR\QuestionnaireResponse\FhirQuestionnaireResponseFormService;
use PHPUnit\Framework\TestCase;

/**
 * Unit tests for FhirQuestionnaireResponseFormService ensuring compliance with US Core 8.0 QuestionnaireResponse Profile
 */
class FhirQuestionnaireResponseFormServiceUnitTest extends TestCase
{
    private FhirQuestionnaireResponseFormService $service;

    protected function setUp(): void
    {
        parent::setUp();
        $this->service = new FhirQuestionnaireResponseFormService();
        $this->service->setSystemLogger(new SystemLogger(Level::Critical));
    }

    /**
     * Test that parseOpenEMRRecord creates a QuestionnaireResponse that conforms to US Core 8.0 Profile
     * Requirements: questionnaire (1..1), status (1..1), subject (1..1), authored (1..1)
     */
    public function testParseOpenEMRRecordReturnsUSCoreCompliantQuestionnaireResponse(): void
    {
        // Generated by Claude.AI - START
        $testUuid = UuidRegistry::uuidToString(UuidRegistry::getRegistryForTable('questionnaire_response')->createUuid());
        $testData = [
            'questionnaire_response_uuid' => $testUuid,
            'questionnaire_foreign_id' => 1,
            'questionnaire_id' => 'test-questionnaire-id',
            'questionnaire_name' => 'Test Questionnaire',
            'patient_id' => 123,
            'puuid' => UuidRegistry::uuidToString(UuidRegistry::getRegistryForTable('patient_data')->createUuid()),
            'encounter' => 456,
            'encounter_uuid' => UuidRegistry::uuidToString(UuidRegistry::getRegistryForTable('form_encounter')->createUuid()),
            'audit_user_id' => 1,
            'creator_user_id' => 1,
            'create_time' => '2025-01-15 10:30:00',
            'last_updated' => '2025-01-15 10:30:00',
            'version' => 1,
            'status' => 'completed',
            'questionnaire' => json_encode([
                'resourceType' => 'Questionnaire',
                'id' => 'test-questionnaire-id',
                'url' => 'http://example.com/questionnaire/test-questionnaire',
                'version' => '1.0.0',
                'name' => 'TestQuestionnaire',
                'title' => 'Test Questionnaire',
                'status' => 'active',
                'item' => []
            ]),
            'questionnaire_response' => json_encode([
                'resourceType' => 'QuestionnaireResponse',
                'id' => $testUuid,
                'questionnaire' => 'http://example.com/questionnaire/test-questionnaire',
                'status' => 'completed',
                'subject' => [
                    'reference' => 'Patient/' . UuidRegistry::uuidToString(UuidRegistry::getRegistryForTable('patient_data')->createUuid())
                ],
                'authored' => '2025-01-15T10:30:00Z',
                'author' => [
                    'reference' => 'Practitioner/test-practitioner-id'
                ],
                'item' => []
            ])
        ];
        // Generated by Claude.AI - END

        $result = $this->service->parseOpenEMRRecord($testData);

        // Test US Core 8.0 required elements
        $this->assertUSCoreQuestionnaireCompliance($result);
        $this->assertUSCoreStatusCompliance($result);
        $this->assertUSCoreSubjectCompliance($result);
        $this->assertUSCoreAuthoredCompliance($result);

        // Test optional but supported US Core elements
        $this->assertUSCoreAuthorCompliance($result);
    }

    /**
     * Test that questionnaire element conforms to US Core requirements
     * Must support: questionnaire (1..1) with optional extensions for display and uri
     */
    public function testUSCoreQuestionnaireElementCompliance(): void
    {
        // Generated by Claude.AI - START
        $testUuid = UuidRegistry::uuidToString(UuidRegistry::getRegistryForTable('questionnaire_response')->createUuid());
        $testData = [
            'questionnaire_response_uuid' => $testUuid,
            'questionnaire_response' => json_encode([
                'resourceType' => 'QuestionnaireResponse',
                'id' => $testUuid,
                'status' => 'completed',
                'subject' => [
                    'reference' => 'Patient/' . UuidRegistry::uuidToString(UuidRegistry::getRegistryForTable('patient_data')->createUuid())
                ],
                'authored' => '2025-01-15T10:30:00Z',
                'questionnaire' => 'http://example.com/questionnaire/test-questionnaire|1.0.0',
                '_questionnaire' => [
                    'extension' => [
                        [
                            'url' => 'http://hl7.org/fhir/StructureDefinition/display',
                            'valueString' => 'Test Questionnaire Display Name'
                        ],
                        [
                            'url' => 'http://hl7.org/fhir/us/core/StructureDefinition/us-core-extension-questionnaire-uri',
                            'valueUri' => 'http://example.com/non-fhir-form'
                        ]
                    ]
                ]
            ])
        ];
        // Generated by Claude.AI - END

        $result = $this->service->parseOpenEMRRecord($testData);

        // Verify questionnaire is present and canonical
        $this->assertNotNull($result->getQuestionnaire());
        $this->assertEquals('http://example.com/questionnaire/test-questionnaire|1.0.0', $result->getQuestionnaire()->getValue());

        // Test that extensions are properly handled if present
        $extensions = $result->getQuestionnaire()->getExtension();
        $this->assertNotEmpty($extensions, "Questionnaire must have the extensions set");
        foreach ($extensions as $extension) {
            $url = $extension->getUrl();
            $this->assertContains($url, [
                'http://hl7.org/fhir/StructureDefinition/display',
                'http://hl7.org/fhir/us/core/StructureDefinition/us-core-extension-questionnaire-uri'
            ]);
        }
    }

    /**
     * Test that items conform to US Core requirements when present
     * Must support: item.linkId (1..1), item.answer.value[x] (0..1), nested items (0..*)
     */
    public function testUSCoreItemsCompliance(): void
    {
        // Generated by Claude.AI - START
        $testUuid = UuidRegistry::uuidToString(UuidRegistry::getRegistryForTable('questionnaire_response')->createUuid());
        $testData = [
            'questionnaire_response_uuid' => $testUuid,
            'questionnaire_response' => json_encode([
                'resourceType' => 'QuestionnaireResponse',
                'id' => $testUuid,
                'questionnaire' => 'http://example.com/questionnaire/test-questionnaire',
                'status' => 'completed',
                'subject' => [
                    'reference' => 'Patient/' . UuidRegistry::uuidToString(UuidRegistry::getRegistryForTable('patient_data')->createUuid())
                ],
                'authored' => '2025-01-15T10:30:00Z',
                'item' => [
                    [
                        'linkId' => '1',
                        'text' => 'First Question',
                        'answer' => [
                            [
                                'valueString' => 'Test Answer'
                            ]
                        ]
                    ],
                    [
                        'linkId' => '2',
                        'text' => 'Second Question',
                        'answer' => [
                            [
                                'valueDecimal' => 42.5
                            ]
                        ],
                        'item' => [
                            [
                                'linkId' => '2.1',
                                'text' => 'Nested Question',
                                'answer' => [
                                    [
                                        'valueCoding' => [
                                            'system' => 'http://loinc.org',
                                            'code' => '72133-2',
                                            'display' => 'Depression severity'
                                        ]
                                    ]
                                ]
                            ]
                        ]
                    ]
                ]
            ])
        ];
        // Generated by Claude.AI - END

        $result = $this->service->parseOpenEMRRecord($testData);

        $items = $result->getItem();
        $this->assertNotEmpty($items, "Items should be present when provided in data");

        // Test first item
        $firstItem = $items[0];
        $this->assertNotNull($firstItem->getLinkId());
        $this->assertEquals('1', $firstItem->getLinkId());
        $this->assertNotEmpty($firstItem->getAnswer());

        // Test second item with nested items
        $secondItem = $items[1];
        $this->assertNotNull($secondItem->getLinkId());
        $this->assertEquals('2', $secondItem->getLinkId());
        $this->assertNotEmpty($secondItem->getItem(), "Nested items should be supported");

        $nestedItem = $secondItem->getItem()[0];
        $this->assertEquals('2.1', $nestedItem->getLinkId());
    }

    /**
     * Test validation of status values against US Core requirements
     * Must be one of: in-progress | completed | amended | entered-in-error | stopped
     */
    public function testUSCoreStatusValidation(): void
    {
        $validStatuses = ['in-progress', 'completed', 'amended', 'entered-in-error', 'stopped'];

        foreach ($validStatuses as $status) {
            // Generated by Claude.AI - START
            $testUuid = UuidRegistry::uuidToString(UuidRegistry::getRegistryForTable('questionnaire_response')->createUuid());
            $testData = [
                'questionnaire_response_uuid' => $testUuid,
                'questionnaire_response' => json_encode([
                    'resourceType' => 'QuestionnaireResponse',
                    'id' => $testUuid,
                    'questionnaire' => 'http://example.com/questionnaire/test-questionnaire',
                    'status' => $status,
                    'subject' => [
                        'reference' => 'Patient/' . UuidRegistry::uuidToString(UuidRegistry::getRegistryForTable('patient_data')->createUuid())
                    ],
                    'authored' => '2025-01-15T10:30:00Z'
                ])
            ];
            // Generated by Claude.AI - END

            $result = $this->service->parseOpenEMRRecord($testData);

            $this->assertNotNull($result->getStatus());
            $this->assertEquals($status, $result->getStatus());
        }
    }

    /**
     * Test that author references conform to US Core requirements
     * Must support US Core Practitioner Profile, but other types are optional
     */
    public function testUSCoreAuthorReferenceCompliance(): void
    {
        // Generated by Claude.AI - START
        $testUuid = UuidRegistry::uuidToString(UuidRegistry::getRegistryForTable('questionnaire_response')->createUuid());
        $testData = [
            'questionnaire_response_uuid' => $testUuid,
            'questionnaire_response' => json_encode([
                'resourceType' => 'QuestionnaireResponse',
                'id' => $testUuid,
                'questionnaire' => 'http://example.com/questionnaire/test-questionnaire',
                'status' => 'completed',
                'subject' => [
                    'reference' => 'Patient/' . UuidRegistry::uuidToString(UuidRegistry::getRegistryForTable('patient_data')->createUuid())
                ],
                'authored' => '2025-01-15T10:30:00Z',
                'author' => [
                    'reference' => 'Practitioner/test-practitioner-id'
                ]
            ])
        ];
        // Generated by Claude.AI - END

        $result = $this->service->parseOpenEMRRecord($testData);

        if ($result->getAuthor()) {
            $authorRef = $result->getAuthor()->getReference();
            $this->assertMatchesRegularExpression('/^(Practitioner|Organization|Patient|PractitionerRole|Device|RelatedPerson)\//', $authorRef);
        }
    }

    /**
     * Test handling of edge cases and error conditions
     */
    public function testEdgeCasesAndErrorHandling(): void
    {
        // Test with minimal valid data - AI Generated comment for minimal test case
        // Generated by Claude.AI - START
        $testUuid = UuidRegistry::uuidToString(UuidRegistry::getRegistryForTable('questionnaire_response')->createUuid());
        $minimalData = [
            'questionnaire_response_uuid' => $testUuid,
            'questionnaire_response' => json_encode([
                'resourceType' => 'QuestionnaireResponse',
                'id' => $testUuid,
                'questionnaire' => 'http://example.com/questionnaire/minimal',
                'status' => 'in-progress',
                'subject' => [
                    'reference' => 'Patient/' . UuidRegistry::uuidToString(UuidRegistry::getRegistryForTable('patient_data')->createUuid())
                ],
                'authored' => '2025-01-15T10:30:00Z'
            ])
        ];
        // Generated by Claude.AI - END

        $result = $this->service->parseOpenEMRRecord($minimalData);
        $this->assertUSCoreMinimalCompliance($result);

        // Test with invalid JSON - should handle gracefully - AI Generated comment for error handling
        $invalidJsonData = [
            'questionnaire_response_uuid' => $testUuid,
            'questionnaire_response' => '{"invalid": json}'
        ];

        $result = $this->service->parseOpenEMRRecord($invalidJsonData);
    }

    // Helper methods to validate US Core compliance - AI Generated helper methods

    /**
     * Assert questionnaire element compliance with US Core 8.0
     * @param FHIRQuestionnaireResponse $response
     */
    private function assertUSCoreQuestionnaireCompliance(FHIRQuestionnaireResponse $response): void
    {
        $this->assertNotNull($response->getQuestionnaire(), "US Core requires questionnaire element (1..1)");
        $this->assertNotEmpty($response->getQuestionnaire(), "Questionnaire canonical URL must not be empty");
    }

    /**
     * Assert status element compliance with US Core 8.0
     * @param FHIRQuestionnaireResponse $response
     */
    private function assertUSCoreStatusCompliance(FHIRQuestionnaireResponse $response): void
    {
        $this->assertNotNull($response->getStatus(), "US Core requires status element (1..1)");
        $validStatuses = ['in-progress', 'completed', 'amended', 'entered-in-error', 'stopped'];
        $this->assertContains($response->getStatus()->getValue(), $validStatuses, "Status must be one of the US Core allowed values");
    }

    /**
     * Assert subject element compliance with US Core 8.0
     * @param FHIRQuestionnaireResponse $response
     */
    private function assertUSCoreSubjectCompliance(FHIRQuestionnaireResponse $response): void
    {
        $this->assertNotNull($response->getSubject(), "US Core requires subject element (1..1)");
        $this->assertNotEmpty($response->getSubject()->getReference(), "Subject reference must not be empty");

        // US Core Profile supports US Core Patient Profile (must support) and Resource (optional)
        $subjectRef = $response->getSubject()->getReference();
        $this->assertMatchesRegularExpression('/^(Patient|Resource)\//', $subjectRef, "Subject must reference Patient or Resource");
    }

    /**
     * Assert authored element compliance with US Core 8.0
     * @param FHIRQuestionnaireResponse $response
     */
    private function assertUSCoreAuthoredCompliance(FHIRQuestionnaireResponse $response): void
    {
        $this->assertNotNull($response->getAuthored(), "US Core requires authored element (1..1)");
        $this->assertNotEmpty($response->getAuthored(), "Authored datetime must not be empty");
    }

    /**
     * Assert author element compliance with US Core 8.0 (optional but supported)
     * @param FHIRQuestionnaireResponse $response
     */
    private function assertUSCoreAuthorCompliance(FHIRQuestionnaireResponse $response): void
    {
        // Author is optional (0..1) but if present must support US Core Practitioner Profile
        if ($response->getAuthor()) {
            $authorRef = $response->getAuthor()->getReference();

            // US Core supports: US Core Practitioner (must support), Organization, Patient, PractitionerRole, Device, RelatedPerson
            $supportedTypes = ['Practitioner', 'Organization', 'Patient', 'PractitionerRole', 'Device', 'RelatedPerson'];
            $matchesAnyType = false;
            foreach ($supportedTypes as $type) {
                if (preg_match('/^' . $type . '\//', $authorRef)) {
                    $matchesAnyType = true;
                    break;
                }
            }
            $this->assertTrue($matchesAnyType, "Author must reference one of the US Core supported resource types");
        }
    }

    /**
     * Assert minimal US Core compliance for edge case testing
     * @param FHIRQuestionnaireResponse $response
     */
    private function assertUSCoreMinimalCompliance(FHIRQuestionnaireResponse $response): void
    {
        $this->assertUSCoreQuestionnaireCompliance($response);
        $this->assertUSCoreStatusCompliance($response);
        $this->assertUSCoreSubjectCompliance($response);
        $this->assertUSCoreAuthoredCompliance($response);
    }

    public function testCreateProvenanceResource(): void
    {
        $this->markTestIncomplete("Not implemented yet");
    }

    public function testSupportsCode(): void
    {
        $loinCodes = ['1234-5', '6789-0', '1111-2', '2222-3'];
        $service = new FhirQuestionnaireFormService();
        foreach ($loinCodes as $code) {
            $this->assertTrue($service->supportsCode($code), "Service should support LOINC code: $code");
        }
    }

    public function testParseOpenEMRRecord(): void
    {
        $service = new FhirQuestionnaireFormService();
        $jsonQuestionnaire = file_get_contents(__DIR__ . '/../../../data/Services/FHIR/Questionnaire/questionnaire-sdc-pathology.json');
        $dataToParse = [
            'questionnaire' => $jsonQuestionnaire
            , 'source_url' => 'http://example.com/source'
            , 'uuid' => 'questionnaire-uuid-123'
        ];
        $parsedQuestionnaire = $service->parseOpenEMRRecord($dataToParse);
        $this->assertEquals($dataToParse['uuid'], $parsedQuestionnaire->getId()->getValue());
        $this->assertEquals('4.0.0', $parsedQuestionnaire->getVersion());
        $this->assertEquals($dataToParse['source_url'], $parsedQuestionnaire->getUrl());
        $this->assertEquals('QuestionnaireSDCProfileExampleCap', $parsedQuestionnaire->getName());
        $this->assertCount(2, $parsedQuestionnaire->getItem());
        $this->assertEquals("1", $parsedQuestionnaire->getItem()[0]->getLinkId());
        $this->assertEquals("2", $parsedQuestionnaire->getItem()[1]->getLinkId());
    }
}
