<?php

/*
 * ObservationServiceTest Tests the ObservationService class
 *
 * @package openemr
 * @link      http://www.open-emr.org
 * @author    Claude.AI on August 21st 2025
 * @author    Stephen Nielson <snielson@discoverandchange.com>
 * @copyright Public Domain for the pieces that were generated by Claude.AI (refactor from interface/forms/observation) view.php,new.php,save.php
 * @license   https://github.com/openemr/openemr/blob/master/LICENSE GNU General Public License 3
 */

namespace OpenEMR\Tests\Services;

use OpenEMR\Common\Database\QueryUtils;
use PHPUnit\Framework\TestCase;
use OpenEMR\Services\ObservationService;
use Ramsey\Uuid\Rfc4122\UuidV4;

// AI Generated Test Class for ObservationService
class ObservationServiceTest extends TestCase
{
    private ObservationService $service;

    private array $sessionBackup;

    private ?string $backupAttendantType;

    private ?int $backupDateFormat = null;

    protected function setUp(): void
    {
        $this->service = new ObservationService();
        $this->sessionBackup = $_SESSION;
        $this->backupAttendantType = $GLOBALS['attendant_type'] ?? null;
        $this->backupDateFormat = $GLOBALS['date_display_format'] ?? null;
        $_SESSION = [
            'pid' => 1
            ,'encounter' => 1
            ,'authProvider' => 'Default'
            ,'authUser' => 1
            ,'userauthorized' => 1
        ];
        $GLOBALS['date_display_format'] = 0; // Y-m-d H:i:s
        $GLOBALS['attendant_type'] = 'pid';
    }

    protected function tearDown(): void
    {
        parent::tearDown(); // TODO: Change the autogenerated stub
        $_SESSION = $this->sessionBackup;
        if ($this->backupAttendantType !== null) {
            $GLOBALS['attendant_type'] = $this->backupAttendantType;
        } else {
            unset($GLOBALS['attendant_type']);
        }
        if ($this->backupDateFormat !== null) {
            $GLOBALS['date_display_format'] = $this->backupDateFormat;
        } else {
            unset($GLOBALS['date_display_format']);
        }
        QueryUtils::sqlStatementThrowException("DELETE FROM forms WHERE formdir = ? AND form_id IN (select form_id FROM form_observation WHERE code LIKE ?)", ['observation', 'OETest:%']);
        QueryUtils::sqlStatementThrowException("DELETE FROM form_observation WHERE code LIKE ?", ['OETest:%']);
    }


    public function testValidateObservationDataValid(): void
    {
        $validData = [
            'pid' => 1,
            'encounter' => 1,
            'groupname' => 'Default',
            'user' => 'admin',
            'form_id' => 2,
            'code' => 'OETest:123',
            'description' => 'Test Description',
            'date' => '2024-01-01'
        ];

        $errors = $this->service->validateObservationData($validData);

        $this->assertEmpty($errors, implode(', ', $errors));
    }

    public function testValidateObservationDataMissingCode(): void
    {
        $invalidData = [
            'description' => 'Test Description',
            'date' => '2024-01-01'
        ];

        $errors = $this->service->validateObservationData($invalidData);

        $this->assertContains('Code is required', $errors);
    }

    public function testValidateObservationDataMissingDescription(): void
    {
        $invalidData = [
            'code' => 'OETest:123',
            'date' => '2024-01-01'
        ];

        $errors = $this->service->validateObservationData($invalidData);

        $this->assertContains('Description is required', $errors);
    }

    public function testValidateObservationDataMissingDate(): void
    {
        $invalidData = [
            'code' => 'OETest:123',
            'description' => 'Test Description'
        ];

        $errors = $this->service->validateObservationData($invalidData);

        $this->assertContains('Date is required', $errors);
    }

    public function testValidateObservationDataAllMissing(): void
    {
        $invalidData = [];

        $errors = $this->service->validateObservationData($invalidData);

        $this->assertCount(8, $errors);
        $this->assertContains('Code is required', $errors);
        $this->assertContains('Description is required', $errors);
        $this->assertContains('Date is required', $errors);
    }

    public function testValidateObservationDataEmptyStrings(): void
    {
        $invalidData = [
            'code' => '',
            'description' => '',
            'date' => ''
        ];

        $errors = $this->service->validateObservationData($invalidData);

        $this->assertCount(8, $errors);
        $this->assertContains('Code is required', $errors);
        $this->assertContains('Description is required', $errors);
        $this->assertContains('Date is required', $errors);
    }

    // end of AI Generated Test Class for ObservationService

    /**
     * Test that would require database mocking - placeholder for integration tests
     */
    public function testGetObservationsByFormId(): void
    {
        $this->markTestIncomplete("This test requires database mocking or integration test setup.");
    }

    public function testGetNextFormId(): void
    {
        $this->markTestIncomplete("This test requires database mocking or integration test setup.");
    }

    public function testDeleteObservationsByFormId(): void
    {
        $this->markTestIncomplete("This test requires database mocking or integration test setup.");
    }

    public function testSaveObservation(): void
    {
        $pid = 1;
        $encounter = 1;
        $observation = [
            'pid' => $pid,
            'encounter' => $encounter,
            'groupname' => 'Default',
            'user' => 'admin',
            'code' => 'OETest:123',
            'ob_value' => 'test_value',
            'ob_unit' => 'mg'
        ];
        // TODO: should we create a test patient for this?
        $savedObservation = $this->service->saveObservation($observation);
        $this->assertNotEmpty($savedObservation, "Failed to save observation");
        $observation = QueryUtils::fetchRecords("select * from form_observation where id = ?", [$savedObservation['id']], true);
        $this->assertNotEmpty($observation, "Failed to create observation record");
        $form = QueryUtils::fetchRecords("select * from forms where form_id = ?", [$observation[0]['form_id']], true);
        $this->assertNotEmpty($form, "Failed to create form record");
    }

    public function testSaveObservationWithSubObservations(): void
    {
        $pid = 1;
        $encounter = 1;
        $observation = [
            'pid' => $pid,
            'encounter' => $encounter,
            'groupname' => 'Default',
            'user' => 'admin',
            'code' => 'OETest:123',
            'ob_value' => 'test_value',
            'ob_unit' => 'mg',
            'sub_observations' => [
                [
                    'pid' => $pid,
                    'encounter' => $encounter,
                    'groupname' => 'Default',
                    'user' => 'admin',
                    'code' => 'OETest:1234',
                    'ob_value' => 'sub_value_1',
                    'ob_unit' => 'ml'
                ],
                [
                    'pid' => $pid,
                    'encounter' => $encounter,
                    'groupname' => 'Default',
                    'user' => 'admin',
                    'code' => 'OETest:1235',
                    'ob_value' => 'sub_value_2',
                    'ob_unit' => 'g'
                ]
            ]
        ];
        // TODO: should we create a test patient for this?
        $savedObservation = $this->service->saveObservation($observation);
        $this->assertNotEmpty($savedObservation, "Failed to save observation");
        $observations = QueryUtils::fetchRecords("select * from form_observation where id = ?", [$savedObservation['id']], true);
        $this->assertNotEmpty($observations, "Failed to create observation record");
        $this->assertObservationEquals($savedObservation, $observations[0]);
        $form = QueryUtils::fetchRecords("select * from forms where form_id = ?", [$savedObservation['form_id']], true);
        $this->assertNotEmpty($form, "Failed to create form record");

        $this->assertCount(2, $savedObservation['sub_observations'], "Failed to save sub-observations");
        foreach ($savedObservation['sub_observations'] as $subObs) {
            $subObservations = QueryUtils::fetchRecords("select * from form_observation where id = ?", [$subObs['id']], true);
            $this->assertNotEmpty($subObservations, "Failed to create sub-observation record");
            $this->assertObservationEquals($subObservations[0], $subObs);
        }
    }

    private function assertObservationEquals(array $expected, array $actual): void
    {
        $this->assertEquals($expected['pid'], $actual['pid']);
        $this->assertEquals($expected['encounter'], $actual['encounter']);
        $this->assertEquals($expected['form_id'], $actual['form_id']);
        $this->assertEquals($expected['code'], $actual['code']);
        $this->assertEquals($expected['ob_value'], $actual['ob_value']);
        $this->assertEquals($expected['ob_unit'], $actual['ob_unit']);
    }

    public function testGetObservationTypes(): void
    {
        // This test would require database mocking
        $this->markTestIncomplete("This test requires database mocking or integration test setup.");
    }

    public function testsearchAndPopulateChildObservationsWithNoDataReturnsEmptyResult(): void
    {
        $uuid = UuidV4::uuid4()->toString();
        // non-existent uuid should return empty result
        $result = $this->service->searchAndPopulateChildObservations(['uuid' => $uuid]);
        $this->assertFalse($result->hasData());
        $this->assertFalse($result->hasErrors());
    }
}
