<?php

/**
 * Unit tests for IdTokenSMARTResponse class handling SMART on FHIR OAuth2 token responses.
 *
 * AI-GENERATED CODE: This test class was generated using Claude AI assistant on August 15, 2025
 * to provide comprehensive unit test coverage for the IdTokenSMARTResponse class, including mocking
 * of dependencies, reflection-based testing of private methods, and various OAuth2 scenarios.
 *
 * This code is released into the public domain as it was entirely generated by Claude AI.
 *
 * @package   OpenEMR
 * @link      http://www.open-emr.org
 * @author    Stephen Nielson <snielson@discoverandchange.com> (AI-assisted)
 * @license   https://github.com/openemr/openemr/blob/master/LICENSE GNU General Public License 3
 */

namespace OpenEMR\Tests\Unit\Common\Auth\OpenIDConnect;

use League\OAuth2\Server\CryptKey;
use League\OAuth2\Server\Entities\AccessTokenEntityInterface;
use League\OAuth2\Server\Entities\ClientEntityInterface;
use League\OAuth2\Server\Entities\ScopeEntityInterface;
use League\OAuth2\Server\Entities\UserEntityInterface;
use Lcobucci\JWT\Token\Builder;
use OpenEMR\Common\Auth\OpenIDConnect\Entities\UserEntity;
use OpenEMR\Common\Auth\OpenIDConnect\IdTokenSMARTResponse;
use OpenEMR\Common\Auth\OpenIDConnect\SMARTSessionTokenContextBuilder;
use OpenEMR\Common\Logging\SystemLogger;
use OpenEMR\Core\OEGlobalsBag;
use OpenIDConnectServer\ClaimExtractor;
use OpenIDConnectServer\Repositories\IdentityProviderInterface;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\StreamInterface;
use Symfony\Component\HttpFoundation\Session\SessionInterface;

class IdTokenSMARTResponseTest extends TestCase
{
    private IdTokenSMARTResponse $idTokenResponse;
    private MockObject $globalsBag;
    private MockObject $session;
    private MockObject $identityProvider;
    private MockObject $claimExtractor;
    private MockObject $logger;

    const KEY_PATH_PRIVATE = __DIR__ . '/../../../../data/Unit/Common/Auth/Grant/openemr-rsa384-private.key';

    protected function setUp(): void
    {
        $mockUserEntity = $this->createMock(UserEntity::class);
        $mockUserEntity->method('getIdentifier')
            ->willReturn('test-user-id');
        $mockUserEntity->method('getClaims')
            ->willReturn([]);
        $this->identityProvider = $this->createMock(IdentityProviderInterface::class);
        $this->identityProvider->method('getUserEntityByIdentifier')
            ->willReturn($mockUserEntity);
        $this->globalsBag = $this->createMock(OEGlobalsBag::class);
        $this->session = $this->createMock(SessionInterface::class);


        $this->claimExtractor = $this->createMock(ClaimExtractor::class);
        $this->claimExtractor->method('extract')
            ->willReturn([]);
        $this->logger = $this->createMock(SystemLogger::class);

        $this->idTokenResponse = new IdTokenSMARTResponse(
            $this->globalsBag,
            $this->session,
            $this->identityProvider,
            $this->claimExtractor,
            $this->createMock(SMARTSessionTokenContextBuilder::class)
        );
        $cryptKey = $this->createMock(CryptKey::class);
        $cryptKey->method('getKeyPath')
            ->willReturn(self::KEY_PATH_PRIVATE);

        $this->idTokenResponse->setPrivateKey($cryptKey);
        $this->idTokenResponse->setSystemLogger($this->logger);
    }

    public function testMarkIsAuthorizationGrant(): void
    {
        // This method doesn't return anything, but we can test it doesn't throw
        $this->idTokenResponse->markIsAuthorizationGrant();
        $this->expectNotToPerformAssertions();
    }

    public function testGenerateHttpResponseWithOfflineAccess(): void
    {
        $accessToken = $this->createMockAccessToken();
        $response = $this->createMockResponse();

        // Mock scope with offline_access
        $scopes = [$this->createMockScope(IdTokenSMARTResponse::SCOPE_OFFLINE_ACCESS)];
        $accessToken->method('getScopes')->willReturn($scopes);

        // Set the access token
        $this->setAccessToken($accessToken);
        $result = $this->idTokenResponse->generateHttpResponse($response);

        $this->assertInstanceOf(ResponseInterface::class, $result);
    }

    public function testGenerateHttpResponseWithoutOfflineAccess(): void
    {
        $accessToken = $this->createMockAccessToken();
        $response = $this->createMockResponse();

        // Mock scopes without offline_access
        $scopes = [$this->createMockScope('openid')];
        $accessToken->method('getScopes')->willReturn($scopes);

        // Mock expiry time
        $expiryTime = new \DateTimeImmutable('+1 hour');
        $accessToken->method('getExpiryDateTime')->willReturn($expiryTime);
        $accessToken->method('__toString')->willReturn('mock_access_token');

        // Set the access token
        $this->setAccessToken($accessToken);

        // Mock the getExtraParams method behavior
        $this->session->method('get')->with('site_id')->willReturn('default');

        $result = $this->idTokenResponse->generateHttpResponse($response);

        $this->assertInstanceOf(ResponseInterface::class, $result);
    }

    public function testSetContextForNewTokens(): void
    {
        $context = ['patient' => 'test-patient-id'];

        $this->idTokenResponse->setContextForNewTokens($context);

        // Since the method doesn't return anything, we test it doesn't throw
        $this->expectNotToPerformAssertions();
    }

    public function testSetContextForNewTokensWithEmptyArray(): void
    {
        $this->idTokenResponse->setContextForNewTokens([]);

        // Should not throw with empty array
        $this->expectNotToPerformAssertions();
    }

    public function testGetBuilderWithNonce(): void
    {
        $accessToken = $this->createMockAccessToken();
        $userEntity = $this->createMockUserEntity();

        $this->globalsBag->method('get')
            ->willReturnMap([
                ['site_addr_oath', 'https://example.com'],
                ['webroot', '/openemr']
            ]);

        $this->session->method('has')->with('nonce')->willReturn(true);
        $this->session->method('get')
            ->willReturnMap([
                ['nonce', 'test-nonce'],
                ['site_id', 'default']
            ]);

        $this->logger->expects($this->once())
            ->method('debug')
            ->with(
                'IdTokenSMARTResponse->getBuilder() nonce found in session',
                ['nonce' => 'test-nonce']
            );

        // Use reflection to test protected method
        $reflection = new \ReflectionClass($this->idTokenResponse);
        $method = $reflection->getMethod('getBuilder');

        $builder = $method->invoke($this->idTokenResponse, $accessToken, $userEntity);

        $this->assertInstanceOf(Builder::class, $builder);
    }

    public function testGetBuilderWithoutNonce(): void
    {
        $accessToken = $this->createMockAccessToken();
        $userEntity = $this->createMockUserEntity();

        $this->globalsBag->method('get')
            ->willReturnMap([
                ['site_addr_oath', 'https://example.com'],
                ['webroot', '/openemr']
            ]);

        $this->session->method('has')->with('nonce')->willReturn(false);
        $this->session->method('get')->with('site_id')->willReturn('default');

        $this->logger->expects($this->once())
            ->method('debug')
            ->with('IdTokenSMARTResponse->getBuilder() no nonce found in session');

        // Use reflection to test protected method
        $reflection = new \ReflectionClass($this->idTokenResponse);
        $method = $reflection->getMethod('getBuilder');

        $builder = $method->invoke($this->idTokenResponse, $accessToken, $userEntity);

        $this->assertInstanceOf(Builder::class, $builder);
    }

    public function testGetExtraParams(): void
    {
        $accessToken = $this->createMockAccessToken();
        $scopes = [
            $this->createMockScope('openid'),
            $this->createMockScope('patient/*.read')
        ];
        $accessToken->method('getScopes')->willReturn($scopes);

        // Mock the context builder behavior through session
        $this->session->method('get')->willReturn(null);

        // Use reflection to test protected method
        $reflection = new \ReflectionClass($this->idTokenResponse);
        $method = $reflection->getMethod('getExtraParams');

        $params = $method->invoke($this->idTokenResponse, $accessToken);

        $this->assertIsArray($params);
        $this->assertArrayHasKey('scope', $params);
        $this->assertEquals('openid patient/*.read', $params['scope']);
    }

    public function testGetExtraParamsWithCustomScopeFiltering(): void
    {
        $accessToken = $this->createMockAccessToken();
        $scopes = [
            $this->createMockScope('openid'),
            $this->createMockScope('patient/*.read'),
            $this->createMockScope('api:oemr') // Should be filtered out
        ];
        $accessToken->method('getScopes')->willReturn($scopes);

        $this->session->method('get')->willReturn(null);

        // Use reflection to test protected method
        $reflection = new \ReflectionClass($this->idTokenResponse);
        $method = $reflection->getMethod('getExtraParams');

        $params = $method->invoke($this->idTokenResponse, $accessToken);

        $this->assertIsArray($params);
        $this->assertArrayHasKey('scope', $params);
        // site:default should be filtered out
        $this->assertEquals('openid patient/*.read', $params['scope']);
    }

    private function createMockAccessToken(): MockObject
    {
        $accessToken = $this->createMock(AccessTokenEntityInterface::class);
        $client = $this->createMock(ClientEntityInterface::class);
        $client->method('getIdentifier')->willReturn('test-client-id');
        $accessToken->method('getClient')->willReturn($client);
        $accessToken->method('getUserIdentifier')->willReturn('test-user-id');
        $accessToken->method('getExpiryDateTime')->willReturn(new \DateTimeImmutable('+1 hour'));

        return $accessToken;
    }

    private function createMockUserEntity(): MockObject
    {
        $userEntity = $this->createMock(UserEntityInterface::class);
        $userEntity->method('getIdentifier')->willReturn('test-user-id');

        return $userEntity;
    }

    private function createMockScope(string $identifier): MockObject
    {
        $scope = $this->createMock(ScopeEntityInterface::class);
        $scope->method('getIdentifier')->willReturn($identifier);

        return $scope;
    }

    private function createMockResponse(): MockObject
    {
        $response = $this->createMock(ResponseInterface::class);
        $stream = $this->createMock(StreamInterface::class);

        $response->method('withStatus')->willReturnSelf();
        $response->method('withHeader')->willReturnSelf();
        $response->method('withBody')->willReturnSelf();
        $response->method('getBody')->willReturn($stream);

        return $response;
    }

    private function setAccessToken(AccessTokenEntityInterface $accessToken): void
    {
        // Use reflection to set the protected accessToken property
        $reflection = new \ReflectionClass($this->idTokenResponse);
        $property = $reflection->getProperty('accessToken');
        $property->setValue($this->idTokenResponse, $accessToken);
    }

    public function testHasScopePrivateMethod(): void
    {
        $scopes = [
            $this->createMockScope('openid'),
            $this->createMockScope('patient/*.read')
        ];

        // Use reflection to test private method
        $reflection = new \ReflectionClass($this->idTokenResponse);
        $method = $reflection->getMethod('hasScope');

        $hasOpenId = $method->invoke($this->idTokenResponse, $scopes, 'openid');
        $hasPatient = $method->invoke($this->idTokenResponse, $scopes, 'patient/*.read');
        $hasLaunch = $method->invoke($this->idTokenResponse, $scopes, 'launch');

        $this->assertTrue($hasOpenId);
        $this->assertTrue($hasPatient);
        $this->assertFalse($hasLaunch);
    }

    public function testGetScopeStringPrivateMethod(): void
    {
        $scopes = [
            $this->createMockScope('openid'),
            $this->createMockScope('patient/*.read'),
            $this->createMockScope('api:oemr'), // Should be filtered out
            $this->createMockScope('offline_access')
        ];

        // Use reflection to test private method
        $reflection = new \ReflectionClass($this->idTokenResponse);
        $method = $reflection->getMethod('getScopeString');

        $scopeString = $method->invoke($this->idTokenResponse, $scopes);

        $this->assertEquals('openid patient/*.read offline_access', $scopeString);
    }
}
