<?php

/**
 * ClientAdminControllerTest - Unit tests for the ClientAdminController class
 * @package openemr
 * @link      http://www.open-emr.org
 * @author    Claude AI
 * @copyright Public Domain, this class was generated by Claude AI and had modifications added afterwards.
 * @license   https://github.com/openemr/openemr/blob/master/LICENSE GNU General Public License 3
 */

namespace OpenEMR\Tests\Unit\FHIR\SMART;

use OpenEMR\Common\Csrf\CsrfUtils;
use OpenEMR\Core\OEGlobalsBag;
use OpenEMR\FHIR\SMART\ExternalClinicalDecisionSupport\RouteController;
use PHPUnit\Framework\TestCase;
use PHPUnit\Framework\MockObject\MockObject;
use OpenEMR\FHIR\SMART\ClientAdminController;
use OpenEMR\Common\Auth\OpenIDConnect\Repositories\ClientRepository;
use OpenEMR\Common\Auth\OpenIDConnect\Entities\ClientEntity;
use OpenEMR\Core\Kernel;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\Session\SessionInterface;
use Symfony\Component\HttpFoundation\Session\Session;
use Symfony\Component\HttpFoundation\Session\Storage\MockArraySessionStorage;
use Symfony\Component\EventDispatcher\EventDispatcher;
use Twig\Environment;

class ClientAdminControllerTest extends TestCase
{
    private OEGlobalsBag $globalsBag;
    private ClientAdminController $controller;
    private MockObject $mockClientRepo;
    private MockObject $mockKernel;
    private SessionInterface $session;
    private string $actionURL = 'admin-client.php';

    protected function setUp(): void
    {
        parent::setUp();

        // Mock dependencies
        $this->mockClientRepo = $this->createMock(ClientRepository::class);
        $this->mockKernel = $this->createMock(Kernel::class);
        $this->session = new Session(new MockArraySessionStorage());
        $this->session->start();
        $this->globalsBag = new OEGlobalsBag([
            'kernel' => $this->mockKernel,
            'web_root' => '',
        ]);
        CsrfUtils::setupCsrfKey($this->session);

        // Mock Twig environment
        $mockTwig = $this->createMock(Environment::class);
        $mockTwig->method('render')->willReturn('<html>Mock Template</html>');

        // Mock event dispatcher
        $mockEventDispatcher = new EventDispatcher();
        $this->mockKernel->method('getEventDispatcher')->willReturn($mockEventDispatcher);

        // Create controller instance
        $this->controller = new ClientAdminController(
            $this->globalsBag,
            $this->session,
            $this->mockClientRepo,
            $this->actionURL,
        );
        $this->controller->setTwig($mockTwig);

        // Set up global variables that might be needed
        $GLOBALS['oauth_ehr_launch_authorization_flow_skip'] = '1';
        $GLOBALS['webroot'] = '/openemr';
    }

    public function testListActionReturnsResponseWithClients(): void
    {
        // Arrange
        $mockClient = $this->createMockClientEntity();
        $this->mockClientRepo->method('listClientEntities')
            ->willReturn([$mockClient]);

        // Act
        $response = $this->controller->listAction();

        // Assert
        $this->assertInstanceOf(Response::class, $response);
        $this->assertEquals(Response::HTTP_OK, $response->getStatusCode());
        $this->assertStringContainsString('Mock Template', $response->getContent());
    }

    public function testEditActionWithValidClientId(): void
    {
        // Arrange
        $clientId = 'test-client-123';
        $mockClient = $this->createMockClientEntity($clientId);
        $request = new Request();

        $this->mockClientRepo->method('getClientEntity')
            ->with($clientId)
            ->willReturn($mockClient);

        // Act
        $response = $this->controller->editAction($clientId, $request);

        // Assert
        $this->assertInstanceOf(Response::class, $response);
        $this->assertEquals(Response::HTTP_OK, $response->getStatusCode());
    }

    public function testEditActionWithInvalidClientIdReturnsNotFound(): void
    {
        // Arrange
        $clientId = 'invalid-client';
        $request = new Request();

        $this->mockClientRepo->method('getClientEntity')
            ->with($clientId)
            ->willReturn(false);

        // Act
        $response = $this->controller->editAction($clientId, $request);

        // Assert
        $this->assertEquals(Response::HTTP_NOT_FOUND, $response->getStatusCode());
    }

    public function testEnableActionSuccessfully(): void
    {
        // Arrange
        $clientId = 'test-client-123';
        $mockClient = $this->createMockClientEntity($clientId);

        $this->mockClientRepo->method('getClientEntity')
            ->with($clientId)
            ->willReturn($mockClient);

        $this->mockClientRepo->method('saveIsEnabled')
            ->with($mockClient, true)
            ->willReturn(true);

        // Act
        $response = $this->controller->enableAction($clientId);

        // Assert
        $this->assertEquals(Response::HTTP_TEMPORARY_REDIRECT, $response->getStatusCode());
        $this->assertStringContainsString(urlencode('edit/' . $clientId), $response->headers->get('Location'));
    }

    public function testDisableActionSuccessfully(): void
    {
        // Arrange
        $clientId = 'test-client-123';
        $mockClient = $this->createMockClientEntity($clientId);

        $this->mockClientRepo->method('getClientEntity')
            ->with($clientId)
            ->willReturn($mockClient);

        $this->mockClientRepo->method('saveIsEnabled')
            ->with($mockClient, false)
            ->willReturn(true);

        // Act
        $response = $this->controller->disableAction($clientId);

        // Assert
        $this->assertEquals(Response::HTTP_TEMPORARY_REDIRECT, $response->getStatusCode());
        $this->assertStringContainsString(urlencode('edit/' . $clientId), $response->headers->get('Location'));
    }

    public function testDispatchWithListAction(): void
    {
        // Arrange
        $request = new Request(['action' => 'list']);
        $this->mockClientRepo->method('listClientEntities')->willReturn([]);

        // Mock the security check method by extending the controller
        $controller = $this->getAdminController();

        // Act
        $response = $controller->dispatch($request);

        // Assert
        $this->assertInstanceOf(Response::class, $response);
        $this->assertEquals(Response::HTTP_OK, $response->getStatusCode());
    }

    public function testDispatchWithEditAction(): void
    {
        // Arrange
        $clientId = 'test-client-123';
        $request = new Request(['action' => "edit/{$clientId}"]);
        $mockClient = $this->createMockClientEntity($clientId);

        $this->mockClientRepo->method('getClientEntity')
            ->with($clientId)
            ->willReturn($mockClient);

        // Mock the security check method
        $controller = $this->getAdminController();

        // Act
        $response = $controller->dispatch($request);

        // Assert
        $this->assertInstanceOf(Response::class, $response);
        $this->assertEquals(Response::HTTP_OK, $response->getStatusCode());
    }

    public function testDispatchWithTokenToolsAction(): void
    {
        // Arrange
        $request = new Request(['action' => ClientAdminController::TOKEN_TOOLS_ACTION]);

        // Mock the security check method
        $controller = $this->getAdminController();

        // Act
        $response = $controller->dispatch($request);

        // Assert
        $this->assertInstanceOf(Response::class, $response);
        $this->assertEquals(Response::HTTP_OK, $response->getStatusCode());
    }

    public function testDispatchWithInvalidActionReturnsNotFound(): void
    {
        // Arrange
        $request = new Request(['action' => 'invalid-action']);

        // Mock the security check method
        $controller = $this->getAdminController();

        // Act
        $response = $controller->dispatch($request);

        // Assert
        $this->assertEquals(Response::HTTP_NOT_FOUND, $response->getStatusCode());
    }

    public function testNotFoundAction(): void
    {
        // Act
        $response = $this->controller->notFoundAction();

        // Assert
        $this->assertEquals(Response::HTTP_NOT_FOUND, $response->getStatusCode());
    }

    public function testEnableAuthorizationFlowSkipAction(): void
    {
        // Arrange
        $clientId = 'test-client-123';
        $mockClient = $this->createMockClientEntity($clientId);

        $this->mockClientRepo->method('getClientEntity')
            ->with($clientId)
            ->willReturn($mockClient);

        $this->mockClientRepo->method('saveSkipEHRLaunchFlow')
            ->with($mockClient, true)
            ->willReturn(true);

        // Use reflection to call private method
        $reflection = new \ReflectionClass($this->controller);
        $method = $reflection->getMethod('enableAuthorizationFlowSkipAction');

        // Act
        $response = $method->invoke($this->controller, $clientId);

        // Assert
        $this->assertEquals(Response::HTTP_TEMPORARY_REDIRECT, $response->getStatusCode());
    }

    public function testDisableAuthorizationFlowSkipAction(): void
    {
        // Arrange
        $clientId = 'test-client-123';
        $mockClient = $this->createMockClientEntity($clientId);

        $this->mockClientRepo->method('getClientEntity')
            ->with($clientId)
            ->willReturn($mockClient);

        $this->mockClientRepo->method('saveSkipEHRLaunchFlow')
            ->with($mockClient, false)
            ->willReturn(true);

        // Use reflection to call private method
        $reflection = new \ReflectionClass($this->controller);
        $method = $reflection->getMethod('disableAuthorizationFlowSkipAction');

        // Act
        $response = $method->invoke($this->controller, $clientId);

        // Assert
        $this->assertEquals(Response::HTTP_TEMPORARY_REDIRECT, $response->getStatusCode());
    }

    public function testGetExternalCDRController(): void
    {
        // Act
        $cdrController = $this->controller->getExternalCDRController();

        // Assert
        $this->assertInstanceOf(\OpenEMR\FHIR\SMART\ExternalClinicalDecisionSupport\RouteController::class, $cdrController);
    }

    public function testGetTwig(): void
    {
        // Act
        $twig = $this->controller->getTwig();

        // Assert
        $this->assertInstanceOf(Environment::class, $twig);
    }

    public function testTokenToolsActionRendersCorrectTemplate(): void
    {
        // Arrange
        $request = new Request();

        // Use reflection to call private method
        $reflection = new \ReflectionClass($this->controller);
        $method = $reflection->getMethod('tokenToolsAction');

        // Act
        $response = $method->invoke($this->controller, $request);

        // Assert
        $this->assertInstanceOf(Response::class, $response);
        $this->assertEquals(Response::HTTP_OK, $response->getStatusCode());
    }

    /**
     * Test data provider for various action scenarios
     */
    public static function actionDataProvider(): array
    {
        return [
            'list action' => ['list', Response::HTTP_OK],
            'empty action defaults to list' => ['', Response::HTTP_OK],
            'token tools action' => [ClientAdminController::TOKEN_TOOLS_ACTION, Response::HTTP_OK],
        ];
    }

    /**
     * @dataProvider actionDataProvider
     */
    public function testDispatchWithVariousActions(string $action, int $expectedStatusCode): void
    {
        // Arrange
        $request = new Request(['action' => $action]);
        $this->mockClientRepo->method('listClientEntities')->willReturn([]);

        // Mock the security check method
        $controller = $this->getAdminController();

        // Act
        $response = $controller->dispatch($request);

        // Assert
        $this->assertEquals($expectedStatusCode, $response->getStatusCode());
    }

    /**
     * Helper method to create a mock ClientEntity
     */
    private function createMockClientEntity(string $identifier = 'test-client'): MockObject|ClientEntity
    {
        $mockClient = $this->createMock(ClientEntity::class);
        $mockClient->method('getIdentifier')->willReturn($identifier);
        $mockClient->method('getName')->willReturn('Test Client');
        $mockClient->method('getContacts')->willReturn(['test@example.com']);
        $mockClient->method('getRegistrationDate')->willReturn('2025-01-01');
        $mockClient->method('isConfidential')->willReturn(true);
        $mockClient->method('isEnabled')->willReturn(true);
        $mockClient->method('shouldSkipEHRLaunchAuthorizationFlow')->willReturn(false);
        $mockClient->method('hasEvidenceDSI')->willReturn(false);
        $mockClient->method('hasPredictiveDSI')->willReturn(false);
        $mockClient->method('hasDSI')->willReturn(false);
        $mockClient->method('getClientRole')->willReturn('client');
        $mockClient->method('getRedirectUri')->willReturn(['https://example.com/callback']);
        $mockClient->method('getLaunchUri')->willReturn('https://example.com/launch');
        $mockClient->method('getJwksUri')->willReturn('https://example.com/jwks');
        $mockClient->method('getJwks')->willReturn('{}');
        $mockClient->method('getScopes')->willReturn(['openid', 'profile']);
        $mockClient->method('getTrustedUsers')->willReturn([]);

        return $mockClient;
    }


    public function testDispatchWithExternalCDR(): void
    {
        // smoke test, individual routes will be handled elsewhere
        $request = new Request(['action' => RouteController::EXTERNAL_CDR_ACTION . '/edit/1']);
        $router = $this->createMock(RouteController::class);
        $router->expects($this->once())->method('dispatch')->willReturn(new Response('', 200));
        $router->expects($this->once())->method("supportsRequest")->willReturn(true);
        $controller = $this->getAdminController();
        $controller->setExternalCDRController($router);
        $response = $controller->dispatch($request);
        $this->assertInstanceOf(Response::class, $response);
        $this->assertEquals(200, $response->getStatusCode());
    }

    protected function tearDown(): void
    {
        parent::tearDown();

        // Clean up any global state
        unset($GLOBALS['oauth_ehr_launch_authorization_flow_skip']);
        unset($GLOBALS['webroot']);
    }

    /**
     * @return ClientAdminController|__anonymous@16704
     */
    private function getAdminController()
    {
        return new class ($this->globalsBag, $this->session, $this->mockClientRepo, $this->actionURL) extends ClientAdminController {
            public function checkSecurity(string $action): void
            {
                // Override to skip ACL check for testing
            }
        };
    }
}
